#include <sys/stat.h>
#include <sys/queue.h>
#include <dirent.h>
#include <err.h>
#include <fcntl.h>
#include <libgen.h>
#include <limits.h>
#include <pthread.h>
#include <openssl/sha.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sqlite3.h>
#include <time.h>
#include <unistd.h>
#include <zlib.h>

#include "delta.h"
#include "rcs-public.h"
#include "xmalloc.h"

#include "common.h"

static pthread_spinlock_t mtx_queue;

static int progress = 1;

static const char create_statement[] =
    "PRAGMA page_size =16384;\n"
    "PRAGMA cache_size=16384;\n"
    "PRAGMA synchronous = OFF;\n"
    "PRAGMA journal_mode = OFF;\n"
    "BEGIN EXCLUSIVE;\n"
    "CREATE TABLE file ("
	"id integer PRIMARY KEY ASC AUTOINCREMENT, "
	"path varchar NOT NULL UNIQUE, "
	"branch varchar NULL, "
	"executable boolean NOT NULL"
    ");\n"
    "CREATE TABLE branches ("
	"id integer PRIMARY KEY ASC AUTOINCREMENT, "
	"symbol varchar NOT NULL UNIQUE, "
	"vendorbranch boolean NOT NULL"
    ");\n"
    "INSERT INTO branches (symbol, vendorbranch) VALUES (\".HEAD\", 0);\n"
    "CREATE TABLE symbol ("
	"id integer PRIMARY KEY ASC AUTOINCREMENT, "
	"file integer NOT NULL REFERENCES file (id), "
	"symbol varchar NOT NULL, "
	"revision varchar NOT NULL, "
	"branch boolean NOT NULL, "
	"UNIQUE (file, symbol)"
    ");\n"
    "CREATE TABLE content ("
	"id integer PRIMARY KEY ASC AUTOINCREMENT, "
	"type integer, "
	"hash varchar UNIQUE NOT NULL, "
	"size integer, "
	"content blob NOT NULL"
    ");\n"
    "CREATE TABLE delta ("
	"hash varchar NOT NULL, "
	"base varchar NOT NULL, "
	"delta blob NOT NULL"
    ");\n"
    "CREATE UNIQUE INDEX delta_hash_base ON delta(hash, base);\n"
    "CREATE TABLE revision ("
	"id INTEGER PRIMARY KEY ASC AUTOINCREMENT, "
	"file integer NOT NULL REFERENCES file (id), "
        "revision varchar NOT NULL, "
	"author varchar NOT NULL, "
	"date datetime NOT NULL, "
	"message varchar NOT NULL, "
	"content NULL REFERENCES content (id), "
	"UNIQUE (file, revision)"
    ");\n"
    "CREATE TABLE revision_link ("
	"parent NOT NULL REFERENCES revision(id), "
	"child NOT NULL REFERENCES revision(id)"
    ");\n"
    "CREATE TABLE revbranches ("
        "branch integer NOT NULL REFERENCES branches(id), "
	"revision integer NOT NULL REFERENCES revision(id) UNIQUE, "
	"file integer NOT NULL REFERENCES file(id)"
    ");\n"
    "CREATE TABLE branchpoints ("
	"branch integer NOT NULL REFERENCES branches(id), "
	"revision integer NULL REFERENCES revision(id), "
	"parent integer NULL REFERENCES branches(id), "
	"file integer NOT NULL REFERENCES file(id)"
    ");\n"
    "CREATE TABLE skiprev ("
	"revision integer NULL REFERENCES revision(id), "
	"file integer NOT NULL REFERENCES file(id)"
    ");\n"
    "CREATE TEMPORARY TABLE symbol2 ("
        "file NOT NULL REFERENCES revision(id), "
	"revision varchar NOT NULL, "
	"branch NOT NULL REFERENCES branch(id)"
    ");\n"
    "CREATE INDEX branchpoints_revision ON branchpoints(revision);\n"
    "CREATE INDEX revision_link_parent ON revision_link(parent);\n"
    "CREATE INDEX revision_date ON revision(date);\n"
    "CREATE INDEX revbranches_revision ON revbranches(revision);\n"
    "CREATE INDEX revbranches_brances ON revbranches(file,branch);\n"
    "CREATE INDEX branches_symbol ON branches(symbol);\n"
    "CREATE INDEX symbol_symbol ON symbol(symbol);\n"
    "CREATE INDEX symbol_revision ON symbol(file,revision);\n";

static const char branchpoint_statement[] =
    "INSERT INTO branchpoints SELECT s.branch, r.id, rb.branch, r.file "
    "FROM revision r, revbranches rb, symbol2 s WHERE "
    "r.revision=s.revision AND r.file=s.file AND rb.revision=r.id AND "
    "NOT EXISTS(SELECT * FROM revbranches rb2 WHERE rb2.branch=s.branch "
    "AND rb2.file=s.file)";

struct cookie {
	TAILQ_ENTRY(cookie) link;
	sqlite3 *db;
	sqlite3_int64 id;
	char *dbpath;
	char *path;

	int branch_id;
	char *branch_rev;
	size_t branch_len;

	sqlite3_stmt *stmt_content;
	sqlite3_stmt *stmt_delta;
	sqlite3_stmt *stmt_revision;
	sqlite3_stmt *stmt_revision2;
	sqlite3_stmt *stmt_revision_link;
	sqlite3_stmt *stmt_file;
	sqlite3_stmt *stmt_head;
	sqlite3_stmt *stmt_symbol;
	sqlite3_stmt *stmt_symbol2;
	sqlite3_stmt *stmt_branch;
	sqlite3_stmt *stmt_vendorbranch;
	sqlite3_stmt *stmt_branch_find;
	sqlite3_stmt *stmt_revbranches;
	sqlite3_stmt *stmt_branching;
	sqlite3_stmt *stmt_branching2;
	sqlite3_stmt *stmt_skiprev;
};

static int threaded;
static TAILQ_HEAD(, cookie) job_queue = TAILQ_HEAD_INITIALIZER(job_queue);

static void
step_and_reset(sqlite3 *db, sqlite3_stmt *stmt)
{
	if (sqlite3_step(stmt) != SQLITE_DONE)
		errx(1, "sqlite3_step failed: %s", sqlite3_errmsg(db));
	if (sqlite3_reset(stmt))
		errx(1, "sqlite3_reset failed: %s", sqlite3_errmsg(db));
}

static char *
compute_digest(const void *data, size_t len)
{
	static const char hex[16] = "0123456789abcdef";

	SHA_CTX ctx;
	unsigned char digest_[SHA_DIGEST_LENGTH];
	char *digest;
	int i;

	SHA1_Init(&ctx);
	SHA1_Update(&ctx, data, len);
	SHA1_Final(digest_, &ctx);
	digest = xmalloc(2 * SHA_DIGEST_LENGTH + 1);
	for (i = 0; i < SHA_DIGEST_LENGTH; ++i) {
		digest[2 * i] = hex[digest_[i] / 16];
		digest[2 * i + 1] = hex[digest_[i] % 16];
	}
	digest[2 * i] = '\0';
	return digest;
}

static void
write_content(const void *content, size_t len, const void *last_content,
    size_t last_len, int forward,
    sqlite3 *db, sqlite3_stmt *stmt, sqlite3_stmt *stmt2, sqlite3_stmt *stmt3)
{
	char *digest, *digest2, *delta;
	unsigned char *output;
	uLongf outputlen;
	int delta_len;

	digest = compute_digest(content, len);
	outputlen = len + len / 100 + 12;
	output = xmalloc(outputlen + 4);
	if (compress2(output + 4, &outputlen, content, len, 9) != Z_OK)
		errx(1, "compressing blob failed");
	output[0] = (len >> 24) & 0xff;
	output[1] = (len >> 16) & 0xff;
	output[2] = (len >> 8) & 0xff;
	output[3] = len & 0xff;
	if (sqlite3_bind_text(stmt, 1, digest, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_int64(stmt, 2, len))
		errx(1, "sqlite3_bind_int64 failed");
	if (sqlite3_bind_blob(stmt, 3, output, outputlen + 4, xfree))
		errx(1, "sqlite3_bind_blob failed");
	step_and_reset(db, stmt);
	sqlite3_bind_null(stmt, 1);
	if (sqlite3_bind_text(stmt2, 6, digest, -1, xfree))
		errx(1, "sqlite3_bind_text failed");

	if (len < 50 || last_len < 50)
		return;

	if (forward) {
		const void *tmp;
		size_t tmp_len;
		tmp = content;
		content = last_content;
		last_content = tmp;
		tmp_len = len;
		len = last_len;
		last_len = tmp_len;
	}

	digest = compute_digest(last_content, last_len);
	digest2 = compute_digest(content, len);
	if (strcmp(digest, digest2) == 0) {
		xfree(digest);
		xfree(digest2);
		return;
	}

	delta = xmalloc(len + 60);
	delta_len = delta_create(last_content, last_len, content, len, delta);
	if (len * 3 < delta_len * 4) {
		xfree(delta);
		xfree(digest);
		xfree(digest2);
		return;
	}

	outputlen = delta_len + delta_len / 100 + 12;
	output = xmalloc(outputlen + 4);
	if (compress2(output + 4, &outputlen, (void *)delta, delta_len, 9) != Z_OK)
		errx(1, "compressing blob failed");
	output[0] = (len >> 24) & 0xff;
	xfree(delta);
	output[1] = (len >> 16) & 0xff;
	output[2] = (len >> 8) & 0xff;
	output[3] = len & 0xff;

	if (sqlite3_bind_text(stmt3, 1, digest, -1, xfree))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_text(stmt3, 2, digest2, -1, xfree))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_blob(stmt3, 3, output, outputlen + 4, xfree))
		errx(1, "sqlite3_bind_blob failed");
	step_and_reset(db, stmt3);
}

static void
import_revision(const void *content, size_t len,
    const void *last_content, size_t last_len,
    const char *rev, const char *last_rev, int forward,
    const char *author, const struct tm *date, const char *message,
    void *cookie_)
{
	struct cookie *cookie = cookie_;
	char buf[128];
	char *dot;
	size_t last_rev_len;
	sqlite3_stmt *stmt_revision;

	if (content) {
		stmt_revision = cookie->stmt_revision2;
		write_content(content, len, last_content, last_len, forward,
		    cookie->db, cookie->stmt_content, stmt_revision,
		    cookie->stmt_delta);
	} else {
		stmt_revision = cookie->stmt_revision;
	}

	strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S", date);

	if (sqlite3_bind_int64(stmt_revision, 1, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");

	if (sqlite3_bind_text(stmt_revision, 2, rev, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_bind_text(stmt_revision, 3, author, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_bind_text(stmt_revision, 4, buf, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_bind_text(stmt_revision, 5, message, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	step_and_reset(cookie->db,stmt_revision);
	sqlite3_bind_null(stmt_revision, 2);
	sqlite3_bind_null(stmt_revision, 3);
	sqlite3_bind_null(stmt_revision, 4);
	sqlite3_bind_null(stmt_revision, 5);

	if ((dot = strrchr(rev, '.')) == NULL)
		errx(1, "Invalid RCS revision %s in %s", rev, cookie->dbpath);
	if (cookie->branch_rev == NULL || rev + cookie->branch_len != dot ||
	    strncmp(rev, cookie->branch_rev, cookie->branch_len)) {
		strlcpy(buf, rev, dot - rev + 1);
		if (sqlite3_bind_int64(cookie->stmt_branch_find, 1, cookie->id))
			errx(1, "sqlite3_bind_int64 failed");
		if (sqlite3_bind_text(cookie->stmt_branch_find, 2, buf, -1,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text failed");
		switch (sqlite3_step(cookie->stmt_branch_find)) {
		case SQLITE_ROW:
			cookie->branch_id =
			    sqlite3_column_int(cookie->stmt_branch_find, 0);
			if (cookie->branch_rev)
				xfree(cookie->branch_rev);
			cookie->branch_rev = xstrdup(buf);
			cookie->branch_len = dot - rev;
			break;
		case SQLITE_DONE:
			if (cookie->branch_rev)
				xfree(cookie->branch_rev);
			cookie->branch_rev = NULL;
			break;
		default:
			errx(1, "sqlite3_step failed");
		}
		if (sqlite3_reset(cookie->stmt_branch_find))
			errx(1, "sqlite3_reset failed");
		sqlite3_bind_null(cookie->stmt_branch_find, 2);
	}

	if (cookie->branch_rev) {
		if (sqlite3_bind_int64(cookie->stmt_revbranches, 1,
		    cookie->id))
			errx(1, "sqlite3_bind_int64 failed");
		if (sqlite3_bind_text(cookie->stmt_revbranches, 2,
		    rev, -1, SQLITE_STATIC))
			errx(1, "sqlite3_bind_text failed");
		if (sqlite3_bind_int(cookie->stmt_revbranches, 3,
		    cookie->branch_id))
			errx(1, "sqlite3_bind_int failed");
		step_and_reset(cookie->db, cookie->stmt_revbranches);
		sqlite3_bind_null(cookie->stmt_revbranches, 2);

		if (last_rev == NULL)
			return;
		last_rev_len = strlen(last_rev);
		if (strncmp(last_rev, rev, last_rev_len) == 0 &&
		    rev[last_rev_len] == '.') {
			if (content) {
				if (sqlite3_bind_int64(cookie->stmt_branching,
				    1, cookie->branch_id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_int64(cookie->stmt_branching,
				    2, cookie->id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_text(cookie->stmt_branching,
				    3, last_rev, -1, SQLITE_STATIC))
					errx(1, "sqlite3_bind_text failed");
				step_and_reset(cookie->db, cookie->stmt_branching);
				sqlite3_bind_null(cookie->stmt_branching, 3);
			} else {
				if (sqlite3_bind_int64(cookie->stmt_branching2,
				    1, cookie->branch_id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_int64(cookie->stmt_branching2,
				    2, cookie->id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_text(cookie->stmt_branching2,
				    3, last_rev, -1, SQLITE_STATIC))
					errx(1, "sqlite3_bind_text failed");
				if (sqlite3_bind_int64(cookie->stmt_branching2,
				    4, cookie->id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_text(cookie->stmt_branching2,
				    5, rev, -1, SQLITE_STATIC))
					errx(1, "sqlite3_bind_text failed");
				step_and_reset(cookie->db, cookie->stmt_branching2);
				sqlite3_bind_null(cookie->stmt_branching, 3);
				sqlite3_bind_null(cookie->stmt_branching, 5);

				if (sqlite3_bind_int64(cookie->stmt_skiprev,
				    1, cookie->id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_text(cookie->stmt_skiprev,
				    2, rev, -1, SQLITE_STATIC))
					errx(1, "sqlite3_bind_text failed");
				if (sqlite3_bind_int64(cookie->stmt_skiprev,
				    3, cookie->id))
					errx(1, "sqlite3_bind_int64 failed");
				if (sqlite3_bind_text(cookie->stmt_skiprev,
				    4, last_rev, -1, SQLITE_STATIC))
					errx(1, "sqlite3_bind_text failed");
				step_and_reset(cookie->db, cookie->stmt_skiprev);
				sqlite3_bind_null(cookie->stmt_skiprev, 2);
			}
		}
	}

	if (last_rev == NULL)
		return;

	if (sqlite3_bind_int64(cookie->stmt_revision_link, 1, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");
	if (sqlite3_bind_text(cookie->stmt_revision_link, forward ? 2 : 4,
	    last_rev, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_int64(cookie->stmt_revision_link, 3, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");
	if (sqlite3_bind_text(cookie->stmt_revision_link, forward ? 4 : 2, rev,
	    -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	step_and_reset(cookie->db, cookie->stmt_revision_link);
	sqlite3_bind_null(cookie->stmt_revision_link, 2);
	sqlite3_bind_null(cookie->stmt_revision_link, 4);

}

static void
import_symbol(const char *symbol, const char *rev, void *cookie_)
{
	struct cookie *cookie = cookie_;
	int is_branch;
	const char *i;
	char buf[128], *dot;
	
	for (is_branch = 1, i = rev; *i; ++i)
		is_branch ^= (*i == '.');

	if (sqlite3_bind_int64(cookie->stmt_symbol, 1, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");
	if (sqlite3_bind_text(cookie->stmt_symbol, 2, symbol, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_text(cookie->stmt_symbol, 3, rev, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_int(cookie->stmt_symbol, 4, is_branch))
		errx(1, "sqlite3_bind_int failed");

	step_and_reset(cookie->db, cookie->stmt_symbol);
	sqlite3_bind_null(cookie->stmt_symbol, 2);
	sqlite3_bind_null(cookie->stmt_symbol, 3);

	if (!is_branch)
		return;

	if (sqlite3_bind_text(cookie->stmt_branch, 1, symbol, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	step_and_reset(cookie->db, cookie->stmt_branch);
	sqlite3_bind_null(cookie->stmt_branch, 1);

	strcpy(buf, rev);
	dot = strrchr(buf, '.');
	if (dot == NULL)
		dot = buf;
	*dot = '\0';

	if (sqlite3_bind_int64(cookie->stmt_symbol2, 1, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");
	if (sqlite3_bind_text(cookie->stmt_symbol2, 2, buf, -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	if (sqlite3_bind_text(cookie->stmt_symbol2, 3, symbol, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	step_and_reset(cookie->db, cookie->stmt_symbol2);
	sqlite3_bind_null(cookie->stmt_symbol2, 2);
	sqlite3_bind_null(cookie->stmt_symbol2, 3);

	if (strcmp(rev, "1.1.1"))
		return;

	if (sqlite3_bind_text(cookie->stmt_vendorbranch, 1, symbol, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	step_and_reset(cookie->db, cookie->stmt_vendorbranch);
	sqlite3_bind_null(cookie->stmt_vendorbranch, 1);
}

static void
import_rcsfile_open(const char *defbranch, void *cookie_)
{
	struct cookie *cookie = cookie_;

	if (sqlite3_bind_int(cookie->stmt_file, 1, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");
	if (sqlite3_bind_text(cookie->stmt_file, 2, cookie->dbpath, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");
	if (defbranch) {
		if (sqlite3_bind_text(cookie->stmt_file, 3, defbranch, -1,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text failed");
	} else {
		if (sqlite3_bind_null(cookie->stmt_file, 3))
			errx(1, "sqlite3_bind_null failed");
	}
	sqlite3_bind_int(cookie->stmt_file, 4, access(cookie->path, X_OK) == 0);

	step_and_reset(cookie->db, cookie->stmt_file);
	sqlite3_bind_null(cookie->stmt_file, 2);
	sqlite3_bind_null(cookie->stmt_file, 3);
	if (sqlite3_bind_int(cookie->stmt_head, 1, cookie->id))
		errx(1, "sqlite3_bind_int64 failed");
	step_and_reset(cookie->db, cookie->stmt_head);
}

static int files, cur_file, permil;

static void *
thread_loop(void *db)
{
	struct cookie *cookie;
	sqlite3_stmt *stmt_content;
	sqlite3_stmt *stmt_delta;
	sqlite3_stmt *stmt_revision;
	sqlite3_stmt *stmt_revision2;
	sqlite3_stmt *stmt_revision_link;
	sqlite3_stmt *stmt_file;
	sqlite3_stmt *stmt_head;
	sqlite3_stmt *stmt_symbol;
	sqlite3_stmt *stmt_symbol2;
	sqlite3_stmt *stmt_branch;
	sqlite3_stmt *stmt_vendorbranch;
	sqlite3_stmt *stmt_branch_find;
	sqlite3_stmt *stmt_revbranches;
	sqlite3_stmt *stmt_branching;
	sqlite3_stmt *stmt_branching2;
	sqlite3_stmt *stmt_skiprev;
	const char *errmsg;

	if (sqlite3_prepare(db, "INSERT OR IGNORE INTO content (type,hash,size,"
	    "content) VALUES (0,?,?,?)",
	    -1, &stmt_content, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, "INSERT OR IGNORE INTO delta (hash,base,delta)"
	    "VALUES (?,?,?)",
	    -1, &stmt_delta, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, "INSERT INTO revision (file,revision,"
	    "author,date,message,content) VALUES (?,?,?,?,?,NULL)",
	    -1, &stmt_revision, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, "INSERT INTO revision (file,revision,"
	    "author,date,message,content) SELECT ?,?,?,?,?,id FROM content "
	    "WHERE hash=?",
	    -1, &stmt_revision2, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, "INSERT INTO revision_link "
	    "SELECT p.id, c.id FROM revision p, revision c WHERE "
	    "p.file=? AND p.revision=? AND c.file=? AND c.revision=?",
	    -1, &stmt_revision_link, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO file (id,path,branch,executable) "
	    "VALUES (?,?,?,?)", -1, &stmt_file, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO symbol "
	    "(file,symbol,revision,branch) "
	    "VALUES (?, \".HEAD\", \"1\", 1)", -1, &stmt_head, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO symbol (file,symbol,revision,"
	    "branch) VALUES (?,?,?,?)",
	    -1, &stmt_symbol, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT OR IGNORE INTO symbol2 "
	    "SELECT ?, ?, b.id FROM branches b WHERE b.symbol=?",
	    -1, &stmt_symbol2, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT OR IGNORE INTO branches "
	    "(symbol,vendorbranch) VALUES (?, 0)",
	    -1, &stmt_branch, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "UPDATE branches SET vendorbranch=1 "
	    "WHERE symbol=?", -1, &stmt_vendorbranch, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "SELECT b.id FROM branches b, symbol s "
	    "WHERE b.symbol=s.symbol AND s.file=? AND s.revision=?", -1,
	    &stmt_branch_find, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO revbranches "
	    "SELECT b.id, r.id, r.file FROM branches b, revision r "
	    "WHERE r.file=? AND r.revision=? AND b.id=?",
	    -1, &stmt_revbranches, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO branchpoints "
	    "SELECT ?, r.id, rb.branch, r.file FROM revision r, revbranches rb "
	    "WHERE r.file=? AND r.revision=? AND rb.revision=r.id AND "
	    "r.content NOT NULL", -1, &stmt_branching, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO branchpoints "
	    "SELECT ?, r.id, rb.branch, r.file FROM revision r, revision r2, revbranches rb "
	    "WHERE r.file=? AND r.revision=? AND rb.revision=r.id AND "
	    "r2.file=? AND r2.revision=? AND "
	    "r.content NOT NULL AND r.date<> r2.date", -1, &stmt_branching2, NULL))
		errx(1, "sqlite3_prepare failed");
	if (sqlite3_prepare(db, "INSERT INTO skiprev SELECT r.id, r.file "
	    "FROM revision r, revision r2 WHERE r.file=? AND r.revision=? AND "
	    "r2.file=? AND r2.revision=? AND r.date=r2.date" , -1,
	    &stmt_skiprev, &errmsg))
		errx(1, "sqlite3_prepare failed: %s", errmsg);

	if (threaded)
		pthread_spin_lock(&mtx_queue);
	for (;;) {
		if ((cookie = TAILQ_FIRST(&job_queue)) == NULL) {
			if (threaded)
				pthread_spin_unlock(&mtx_queue);
			break;
		}
		TAILQ_REMOVE(&job_queue, cookie, link);

		if (threaded)
			pthread_spin_unlock(&mtx_queue);

		cookie->stmt_content = stmt_content;
		cookie->stmt_delta = stmt_delta;
		cookie->stmt_revision = stmt_revision;
		cookie->stmt_revision2 = stmt_revision2;
		cookie->stmt_revision_link = stmt_revision_link;
		cookie->stmt_file = stmt_file;
		cookie->stmt_head = stmt_head;
		cookie->stmt_symbol = stmt_symbol;
		cookie->stmt_symbol2 = stmt_symbol2;
		cookie->stmt_branch = stmt_branch;
		cookie->stmt_vendorbranch = stmt_vendorbranch;
		cookie->branch_rev = NULL;
		cookie->stmt_branch_find = stmt_branch_find;
		cookie->stmt_revbranches = stmt_revbranches;
		cookie->stmt_branching = stmt_branching;
		cookie->stmt_branching2 = stmt_branching2;
		cookie->stmt_skiprev = stmt_skiprev;

		if (rcs_getrevisions(cookie->path, import_rcsfile_open,
		    import_symbol, import_revision, cookie))
			errx(1, "parsing rcs file failed");

		if (cookie->branch_rev)
			xfree(cookie->branch_rev);
		xfree(cookie->dbpath);
		xfree(cookie->path);
		xfree(cookie);

		if (threaded)
			pthread_spin_lock(&mtx_queue);

		if (progress) {
			int cur_permil = cur_file++ * 1000 / files;

			if (cur_permil != permil) {
				permil = cur_permil;
				printf("\r%d.%d%%", cur_permil / 10,
				    cur_permil % 10);
				fflush(stdout);
			}
		}
	}

	sqlite3_finalize(stmt_content);
	sqlite3_finalize(stmt_delta);
	sqlite3_finalize(stmt_revision);
	sqlite3_finalize(stmt_revision2);
	sqlite3_finalize(stmt_revision_link);
	sqlite3_finalize(stmt_file);
	sqlite3_finalize(stmt_head);
	sqlite3_finalize(stmt_symbol);
	sqlite3_finalize(stmt_symbol2);
	sqlite3_finalize(stmt_branch);
	sqlite3_finalize(stmt_vendorbranch);
	sqlite3_finalize(stmt_branch_find);
	sqlite3_finalize(stmt_revbranches);
	sqlite3_finalize(stmt_branching);
	sqlite3_finalize(stmt_branching2);
	sqlite3_finalize(stmt_skiprev);

	return NULL;
}

static void
add_job(sqlite3 *db, const char *path, const char *dbpath)
{
	struct cookie *cookie = xmalloc(sizeof(*cookie));

	cookie->id = ++files;
	cookie->db = db;
	cookie->dbpath = xstrdup(dbpath);
	cookie->path = xstrdup(path);

	TAILQ_INSERT_TAIL(&job_queue, cookie, link);
}

static void
parseattic(sqlite3 *db, const char *path, size_t prefix_len)
{
	DIR *d;
	struct dirent *dp;
	char buf[PATH_MAX];
	char buf2[PATH_MAX];
	int len;

	snprintf(buf, sizeof(buf), "%s/Attic", path);
	if ((d = opendir(buf)) == NULL)
		err(1, "directory %s can't be opened", buf);

	while ((dp = readdir(d)) != NULL) {
		if (strcmp(dp->d_name, ".") == 0)
			continue;
		if (strcmp(dp->d_name, "..") == 0)
			continue;
		snprintf(buf, sizeof(buf), "%s/Attic/%s", path, dp->d_name);
		len = snprintf(buf2, sizeof(buf2), "%s/%s", path, dp->d_name);
		if (len < 3 || buf2[len - 2] != ',' || buf2[len - 1] != 'v')
			errx(1, "Non-RCS file in Attic found: %s", buf);
		buf2[len - 2] = 0;
		if (dp->d_type == DT_DIR)
			errx(1, "Directory in the Attic found");
		add_job(db, buf, buf2 + prefix_len);
	}

	closedir(d);
}

static void
parsedir(sqlite3 *db, const char *path, size_t prefix_len)
{
	DIR *d;
	struct dirent *dp;
	struct stat sb;
	char buf[PATH_MAX], buf2[PATH_MAX];
	int len;

	if ((d = opendir(path)) == NULL)
		err(1, "directory %s can't be opened", path);

	while ((dp = readdir(d)) != NULL) {
		if (strcmp(dp->d_name, ".") == 0)
			continue;
		if (strcmp(dp->d_name, "..") == 0)
			continue;
		if (strcmp(dp->d_name, "CVS") == 0)
			continue;
		len = snprintf(buf, sizeof(buf), "%s/%s", path, dp->d_name);
		stat(buf, &sb);
		if (strcmp(dp->d_name, "Attic") == 0) {
			if (!S_ISDIR(sb.st_mode))
				errx(1, "Attic is not directory");
			parseattic(db, path, prefix_len);
		} else if (S_ISDIR(sb.st_mode)) {
			parsedir(db, buf, prefix_len);
		} else if (len < 3 || buf[len - 2] != ',' ||
		    buf[len - 1] != 'v') {
			errx(1, "Non-RCS file found: %s", buf);
		} else {
			snprintf(buf2, sizeof(buf2), "%s/Attic/%s", path,
			    dp->d_name);
			if (access(buf2, F_OK) == 0)
				errx(1, "File also exists in the Attic: %s",
				    buf);
			memcpy(buf2, buf, len - 2);
			buf2[len - 2] = 0;
			add_job(db, buf, buf2 + prefix_len);
		}
	}

	closedir(d);
}

static void
usage(void)
{
	fprintf(stderr, "Usage: %s [-aqs] database directory", getprogname());
	exit(1);
}

int
main(int argc, char **argv)
{
	sqlite3 *db;
	char *errmsg;
	char *c;
	size_t prefix_len;
	long ncpu;
	pthread_t *threads;
	int ch, i, analyze = 0, is_module = 0;

	ncpu = sysconf(_SC_NPROCESSORS_ONLN);
	threaded = ncpu >= 2;

	while ((ch = getopt(argc, argv, "amqs")) != -1) {
		switch (ch) {
		case 'a':
			analyze = 1;
			break;
		case 'm':
			/*
			 * Consider the directory as module and skip
			 * the last component as well.
			 */
			is_module = 1;
			break;
		case 'q':
			progress = 0;
			break;
		case 's':
			threaded = 0;
			break;
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 2)
		usage();

	unlink(argv[0]);

	if (threaded)
		sqlite3_config(SQLITE_CONFIG_SERIALIZED);
	else
		sqlite3_config(SQLITE_CONFIG_SINGLETHREAD);

	if (sqlite3_open_v2(argv[0], &db,
	    SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE, NULL))
		errx(1, "database open failed: %s", sqlite3_errmsg(db));

	if (sqlite3_exec(db, create_statement, NULL, 0, &errmsg))
		errx(1, "create database tables failed: %s", errmsg);

	/* Remove trailing slashes */
	while ((c = strrchr(argv[1], '/')) != NULL && c[1] == '\0')
		c[0] = '\0';
	prefix_len = (c == NULL) ? 0 : c - argv[1] + 1;
	if (is_module)
		prefix_len += strlen(argv[1] + prefix_len) + 1;

	threads = xcalloc(ncpu, sizeof(*threads));

	if (progress) {
		printf("Counting files...");
		fflush(stdout);
	}

	parsedir(db, argv[1], prefix_len);

	if (progress) {
		printf(" %d file%s found\n", files, files == 1 ? "" : "s");
		printf("Importing RCS files...\n0.0%%");
		fflush(stdout);
	}

	if (threaded) {
		pthread_spin_init(&mtx_queue, PTHREAD_PROCESS_SHARED);
		for (i = 0; i < ncpu; ++i)
			pthread_create(threads + i, NULL, thread_loop, db);

		for (i = 0; i < ncpu; ++i)
			pthread_join(threads[i], NULL);
	} else {
		thread_loop(db);
	}

	xfree(threads);

	if (progress) {
		printf("\r...done\n");
		fflush(stdout);
	}

	printf("Adding branch points...\n");
	if (sqlite3_exec(db, branchpoint_statement, NULL, 0, &errmsg))
		errx(1, "command failed: %s", errmsg);

	if (analyze) {
		printf("Rebuilding sqlite statistics...\n");
		sqlite3_exec(db, "analyze", NULL, 0, &errmsg);
	}

	if (sqlite3_exec(db, "COMMIT", NULL, 0, &errmsg))
		errx(1, "commit failed: %s", errmsg);

	if (sqlite3_close(db))
		errx(1, "db close failed");
	return 0;
}
