/*	$OpenBSD: rcs.c,v 1.71 2010/10/27 08:35:45 tobias Exp $	*/
/*
 * Copyright (c) 2003, 2004 Jean-Francois Brousseau <jfb@openbsd.org>
 * Copyright (c) 2005, 2006 Joris Vink <joris@openbsd.org>
 * Copyright (c) 2006 Xavier Santolaria <xsa@openbsd.org>
 * Copyright (c) 2006 Niall O'Higgins <niallo@openbsd.org>
 * Copyright (c) 2006 Ray Lai <ray@openbsd.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL  DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 2010 Tobias Stoeckmann <tobias@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define RCS_REV_BUFSZ	64

#include <sys/param.h>
#include <sys/queue.h>
#include <sys/stat.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <pwd.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "rcs.h"
#include "xmalloc.h"

#include "common.h"

#ifndef MIN
#define	MIN(a,b)	((/*CONSTCOND*/(a)<(b))?(a):(b))
#endif

#define BUF_INCR	128
#define RCS_KWEXP_SIZE  1024

static int rcsparse_deltas(RCSFILE *, RCSNUM *);
static int rcsparse_deltatexts(RCSFILE *, RCSNUM *);
static void rcsparse_free(RCSFILE *);
static int rcsparse_init(RCSFILE *);


/* invalid characters in RCS states */
static const char rcs_state_invch[] = RCS_STATE_INVALCHAR;

/* invalid characters in RCS symbol names */
static const char rcs_sym_invch[] = RCS_SYM_INVALCHAR;

/* comment leaders, depending on the file's suffix */
static const struct rcs_comment {
	const char	*rc_suffix;
	const char	*rc_cstr;
} rcs_comments[] = {
	{ "1",    ".\\\" " },
	{ "2",    ".\\\" " },
	{ "3",    ".\\\" " },
	{ "4",    ".\\\" " },
	{ "5",    ".\\\" " },
	{ "6",    ".\\\" " },
	{ "7",    ".\\\" " },
	{ "8",    ".\\\" " },
	{ "9",    ".\\\" " },
	{ "a",    "-- "    },	/* Ada		 */
	{ "ada",  "-- "    },
	{ "adb",  "-- "    },
	{ "asm",  ";; "    },	/* assembler (MS-DOS) */
	{ "ads",  "-- "    },	/* Ada */
	{ "bat",  ":: "    },	/* batch (MS-DOS) */
	{ "body", "-- "    },	/* Ada */
	{ "c",    " * "    },	/* C */
	{ "c++",  "// "    },	/* C++ */
	{ "cc",   "// "    },
	{ "cpp",  "// "    },
	{ "cxx",  "// "    },
	{ "m",    "// "    },	/* Objective-C */
	{ "cl",   ";;; "   },	/* Common Lisp	 */
	{ "cmd",  ":: "    },	/* command (OS/2) */
	{ "cmf",  "c "     },	/* CM Fortran	 */
	{ "csh",  "# "     },	/* shell	 */
	{ "e",    "# "     },	/* efl		 */
	{ "epsf", "% "     },	/* encapsulated postscript */
	{ "epsi", "% "     },	/* encapsulated postscript */
	{ "el",   "; "     },	/* Emacs Lisp	 */
	{ "f",    "c "     },	/* Fortran	 */
	{ "for",  "c "     },
	{ "h",    " * "    },	/* C-header	 */
	{ "hh",   "// "    },	/* C++ header	 */
	{ "hpp",  "// "    },
	{ "hxx",  "// "    },
	{ "in",   "# "     },	/* for Makefile.in */
	{ "l",    " * "    },	/* lex */
	{ "mac",  ";; "    },	/* macro (DEC-10, MS-DOS, PDP-11, VMS, etc) */
	{ "mak",  "# "     },	/* makefile, e.g. Visual C++ */
	{ "me",   ".\\\" " },	/* me-macros	t/nroff	 */
	{ "ml",   "; "     },	/* mocklisp	 */
	{ "mm",   ".\\\" " },	/* mm-macros	t/nroff	 */
	{ "ms",   ".\\\" " },	/* ms-macros	t/nroff	 */
	{ "man",  ".\\\" " },	/* man-macros	t/nroff	 */
	{ "p",    " * "    },	/* pascal	 */
	{ "pas",  " * "    },
	{ "pl",   "# "     },	/* Perl	(conflict with Prolog) */
	{ "pm",   "# "     },	/* Perl	module */
	{ "ps",   "% "     },	/* postscript */
	{ "psw",  "% "     },	/* postscript wrap */
	{ "pswm", "% "     },	/* postscript wrap */
	{ "r",    "# "     },	/* ratfor	 */
	{ "rc",   " * "    },	/* Microsoft Windows resource file */
	{ "red",  "% "     },	/* psl/rlisp	 */
	{ "sh",   "# "     },	/* shell	 */
	{ "sl",   "% "     },	/* psl		 */
	{ "spec", "-- "    },	/* Ada		 */
	{ "tex",  "% "     },	/* tex		 */
	{ "y",    " * "    },	/* yacc		 */
	{ "ye",   " * "    },	/* yacc-efl	 */
	{ "yr",   " * "    },	/* yacc-ratfor	 */
};

static const struct rcs_kw rcs_expkw[] =  {
	{ "Author",	RCS_KW_AUTHOR   },
	{ "Date",	RCS_KW_DATE     },
	{ "Header",	RCS_KW_HEADER   },
	{ "Id",		RCS_KW_ID       },
	{ "NetBSD",	RCS_KW_ID       },
	{ "Log",	RCS_KW_LOG | RCS_KW_RCSFILE },
	{ "Name",	RCS_KW_NAME     },
	{ "RCSfile",	RCS_KW_RCSFILE  },
	{ "Revision",	RCS_KW_REVISION },
	{ "Source",	RCS_KW_SOURCE   },
	{ "State",	RCS_KW_STATE    },
	{ "Locker",	RCS_KW_LOCKER	},
};

#define NB_COMTYPES	(sizeof(rcs_comments)/sizeof(rcs_comments[0]))

struct rcs_line {
	u_char			*l_line;
	int			 l_lineno;
	size_t			 l_len;
	TAILQ_ENTRY(rcs_line)	 l_list;
};

TAILQ_HEAD(tqh, rcs_line);

struct rcs_lines {
	int		 l_nblines;
	struct tqh	 l_lines;
};

struct buf {
	u_char	*cb_buf;
	size_t	 cb_size;
	size_t	 cb_len;
};

typedef struct buf BUF;

static RCSFILE		*rcs_open(const char *, int);
static void		 rcs_close(RCSFILE *);
static int			 rcs_kwexp_get(RCSFILE *);

static int	rcs_kflag_get(const char *);
static BUF	*rcs_kwexp_buf(BUF *, RCSFILE *, RCSNUM *);

static void rcs_freelines(struct rcs_lines *);
static struct rcs_lines *rcs_splitlines(u_char *, size_t);

static BUF *rcs_patchfile(u_char *, size_t, u_char *, size_t,
    int (*)(struct rcs_lines *, struct rcs_lines *));

static void	rcs_freedelta(struct rcs_delta *);

static size_t
buf_len(BUF *b)
{
	return (b->cb_len);
}

static u_char *
buf_get(BUF *bp)
{
	return (bp->cb_buf);
}

/*
 * buf_grow()
 *
 * Grow the buffer <b> by <len> bytes.  The contents are unchanged by this
 * operation regardless of the result.
 */
static void
buf_grow(BUF *b, size_t len)
{
	b->cb_buf = xrealloc(b->cb_buf, 1, b->cb_size + len);
	b->cb_size += len;
}

static void
buf_append(BUF *b, const void *data, size_t len)
{
	size_t left;
	u_char *bp;

	left = b->cb_size - b->cb_len;

	if (left < len)
		buf_grow(b, len - left);

	bp = b->cb_buf + b->cb_len;
	memcpy(bp, data, len);
	b->cb_len += len;
}

static void
buf_putc(BUF *b, int c)
{
	u_char *bp;

	bp = b->cb_buf + b->cb_len;
	if (bp == (b->cb_buf + b->cb_size)) {
		buf_grow(b, BUF_INCR);
		bp = b->cb_buf + b->cb_len;
	}
	*bp = (u_char)c;
	b->cb_len++;
}

static void
buf_puts(BUF *b, const char *str)
{
	buf_append(b, str, strlen(str));
}

static void
buf_free(BUF *b)
{
	if (b->cb_buf != NULL)
		xfree(b->cb_buf);
	xfree(b);
}

static void *
buf_release(BUF *b)
{
	void *tmp;

	tmp = b->cb_buf;
	xfree(b);
	return (tmp);
}

static BUF *
buf_alloc(size_t len)
{
	BUF *b;

	b = xmalloc(sizeof(*b));
	/* Postpone creation of zero-sized buffers */
	if (len > 0)
		b->cb_buf = xcalloc(1, len);
	else
		b->cb_buf = NULL;

	b->cb_size = len;
	b->cb_len = 0;

	return (b);
}

static RCSFILE *
rcs_open(const char *path, int fd)
{
	mode_t fmode;
	RCSFILE *rfp;
	struct rcs_delta *rdp;
	struct rcs_lock *lkr;

	fmode = S_IRUSR|S_IRGRP|S_IROTH;

	rfp = xcalloc(1, sizeof(*rfp));

	rfp->rf_path = xstrdup(path);
	rfp->rf_flags = RCS_SLOCK | RCS_SYNCED;
	rfp->rf_mode = fmode;
	if (fd == -1)
		rfp->rf_file = NULL;
	else if ((rfp->rf_file = fdopen(fd, "r")) == NULL)
		err(1, "rcs_open: fdopen: `%s'", path);

	TAILQ_INIT(&(rfp->rf_delta));
	TAILQ_INIT(&(rfp->rf_access));
	TAILQ_INIT(&(rfp->rf_symbols));
	TAILQ_INIT(&(rfp->rf_locks));

	if (rcsparse_init(rfp))
		errx(1, "could not parse admin data");

	/* fill in rd_locker */
	TAILQ_FOREACH(lkr, &(rfp->rf_locks), rl_list) {
		if ((rdp = rcs_findrev(rfp, lkr->rl_num)) == NULL) {
			rcs_close(rfp);
			return (NULL);
		}

		rdp->rd_locker = xstrdup(lkr->rl_name);
	}

	return (rfp);
}

/*
 * rcs_close()
 *
 * Close an RCS file handle.
 */
static void
rcs_close(RCSFILE *rfp)
{
	struct rcs_delta *rdp;
	struct rcs_access *rap;
	struct rcs_lock *rlp;
	struct rcs_sym *rsp;

	while (!TAILQ_EMPTY(&(rfp->rf_delta))) {
		rdp = TAILQ_FIRST(&(rfp->rf_delta));
		TAILQ_REMOVE(&(rfp->rf_delta), rdp, rd_list);
		rcs_freedelta(rdp);
	}

	while (!TAILQ_EMPTY(&(rfp->rf_access))) {
		rap = TAILQ_FIRST(&(rfp->rf_access));
		TAILQ_REMOVE(&(rfp->rf_access), rap, ra_list);
		xfree(rap->ra_name);
		xfree(rap);
	}

	while (!TAILQ_EMPTY(&(rfp->rf_symbols))) {
		rsp = TAILQ_FIRST(&(rfp->rf_symbols));
		TAILQ_REMOVE(&(rfp->rf_symbols), rsp, rs_list);
		rcsnum_free(rsp->rs_num);
		xfree(rsp->rs_name);
		xfree(rsp);
	}

	while (!TAILQ_EMPTY(&(rfp->rf_locks))) {
		rlp = TAILQ_FIRST(&(rfp->rf_locks));
		TAILQ_REMOVE(&(rfp->rf_locks), rlp, rl_list);
		rcsnum_free(rlp->rl_num);
		xfree(rlp->rl_name);
		xfree(rlp);
	}

	if (rfp->rf_head != NULL)
		rcsnum_free(rfp->rf_head);
	if (rfp->rf_branch != NULL)
		rcsnum_free(rfp->rf_branch);

	if (rfp->rf_file != NULL)
		fclose(rfp->rf_file);
	if (rfp->rf_path != NULL)
		xfree(rfp->rf_path);
	if (rfp->rf_comment != NULL)
		xfree(rfp->rf_comment);
	if (rfp->rf_expand != NULL)
		xfree(rfp->rf_expand);
	if (rfp->rf_desc != NULL)
		xfree(rfp->rf_desc);
	if (rfp->rf_pdata != NULL)
		rcsparse_free(rfp);
	xfree(rfp);
}

/*
 * rcs_sym_check()
 *
 * Check the RCS symbol name <sym> for any unsupported characters.
 * Returns 1 if the tag is correct, 0 if it isn't valid.
 */
static int
rcs_sym_check(const char *sym)
{
	int ret;
	const char *cp;

	ret = 1;
	cp = sym;
	if (!isalpha((unsigned char)*cp++))
		return (0);

	for (; *cp != '\0'; cp++)
		if (!isgraph((unsigned char)*cp) ||
		    (strchr(rcs_sym_invch, *cp) != NULL)) {
			ret = 0;
			break;
		}

	return (ret);
}

/*
 * rcs_findrev()
 *
 * Find a specific revision's delta entry in the tree of the RCS file <rfp>.
 * The revision number is given in <rev>.
 *
 * If the given revision is a branch number, we translate it into the latest
 * revision on the branch.
 *
 * Returns a pointer to the delta on success, or NULL on failure.
 */
static struct rcs_delta *
rcs_findrev(RCSFILE *rfp, RCSNUM *rev)
{
	u_int cmplen;
	struct rcs_delta *rdp;
	RCSNUM *brev, *frev;

	/*
	 * We need to do more parsing if the last revision in the linked list
	 * is greater than the requested revision.
	 */
	rdp = TAILQ_LAST(&(rfp->rf_delta), rcs_dlist);
	if (rdp == NULL ||
	    rcsnum_cmp(rdp->rd_num, rev, 0) == -1) {
		if (rcsparse_deltas(rfp, rev))
			return (NULL);
	}

	/*
	 * Translate a branch into the latest revision on the branch itself.
	 */
	if (RCSNUM_ISBRANCH(rev)) {
		brev = rcsnum_brtorev(rev);
		frev = brev;
		for (;;) {
			rdp = rcs_findrev(rfp, frev);
			if (rdp == NULL)
				return (NULL);

			if (rdp->rd_next->rn_len == 0)
				break;

			frev = rdp->rd_next;
		}

		rcsnum_free(brev);
		return (rdp);
	}

	cmplen = rev->rn_len;

	TAILQ_FOREACH(rdp, &(rfp->rf_delta), rd_list) {
		if (rcsnum_cmp(rdp->rd_num, rev, cmplen) == 0)
			return (rdp);
	}

	return (NULL);
}

/*
 * rcs_kwexp_get()
 *
 * Retrieve the keyword expansion mode to be used for the RCS file <file>.
 */
static int
rcs_kwexp_get(RCSFILE *file)
{
	if (file->rf_expand == NULL)
		return (RCS_KWEXP_DEFAULT);

	return (rcs_kflag_get(file->rf_expand));
}

/*
 * rcs_kflag_get()
 *
 * Get the keyword expansion mode from a set of character flags given in
 * <flags> and return the appropriate flag mask.  In case of an error, the
 * returned mask will have the RCS_KWEXP_ERR bit set to 1.
 */
static int
rcs_kflag_get(const char *flags)
{
	int fl;
	size_t len;
	const char *fp;

	if (flags == NULL || !(len = strlen(flags)))
		return (RCS_KWEXP_ERR);

	fl = 0;
	for (fp = flags; *fp != '\0'; fp++) {
		if (*fp == 'k')
			fl |= RCS_KWEXP_NAME;
		else if (*fp == 'v')
			fl |= RCS_KWEXP_VAL;
		else if (*fp == 'l')
			fl |= RCS_KWEXP_LKR;
		else if (*fp == 'o') {
			if (len != 1)
				fl |= RCS_KWEXP_ERR;
			fl |= RCS_KWEXP_OLD;
		} else if (*fp == 'b') {
			if (len != 1)
				fl |= RCS_KWEXP_ERR;
			fl |= RCS_KWEXP_NONE;
		} else	/* unknown letter */
			fl |= RCS_KWEXP_ERR;
	}

	return (fl);
}

/*
 * rcs_freedelta()
 *
 * Free the contents of a delta structure.
 */
static void
rcs_freedelta(struct rcs_delta *rdp)
{
	struct rcs_branch *rb;

	if (rdp->rd_num != NULL)
		rcsnum_free(rdp->rd_num);
	if (rdp->rd_next != NULL)
		rcsnum_free(rdp->rd_next);

	if (rdp->rd_author != NULL)
		xfree(rdp->rd_author);
	if (rdp->rd_locker != NULL)
		xfree(rdp->rd_locker);
	if (rdp->rd_state != NULL)
		xfree(rdp->rd_state);
	if (rdp->rd_log != NULL)
		xfree(rdp->rd_log);
	if (rdp->rd_text != NULL)
		xfree(rdp->rd_text);

	while ((rb = TAILQ_FIRST(&(rdp->rd_branches))) != NULL) {
		TAILQ_REMOVE(&(rdp->rd_branches), rb, rb_list);
		rcsnum_free(rb->rb_num);
		xfree(rb);
	}

	xfree(rdp);
}

/*
 * rcs_state_check()
 *
 * Check if string <state> is valid.
 *
 * Returns 0 if the string is valid, -1 otherwise.
 */
static int
rcs_state_check(const char *state)
{
	int ret;
	const char *cp;

	ret = 0;
	cp = state;
	if (!isalpha((unsigned char)*cp++))
		return (-1);

	for (; *cp != '\0'; cp++)
		if (!isgraph((unsigned char)*cp) ||
		    (strchr(rcs_state_invch, *cp) != NULL)) {
			ret = -1;
			break;
		}

	return (ret);
}

static int
rcs_patch_lines(struct rcs_lines *dlines, struct rcs_lines *plines)
{
	char op, *ep;
	struct rcs_line *lp, *dlp, *ndlp;
	int i, lineno, nbln;
	u_char tmp;

	dlp = TAILQ_FIRST(&(dlines->l_lines));
	lp = TAILQ_FIRST(&(plines->l_lines));

	/* skip first bogus line */
	for (lp = TAILQ_NEXT(lp, l_list); lp != NULL;
	    lp = TAILQ_NEXT(lp, l_list)) {
		if (lp->l_len < 2)
			errx(1, "line too short, RCS patch seems broken");
		op = *(lp->l_line);
		/* NUL-terminate line buffer for strtol() safety. */
		tmp = lp->l_line[lp->l_len - 1];
		lp->l_line[lp->l_len - 1] = '\0';
		lineno = (int)strtol((char *)(lp->l_line + 1), &ep, 10);
		if (lineno > dlines->l_nblines || lineno < 0 ||
		    *ep != ' ')
			errx(1, "invalid line specification in RCS patch");
		ep++;
		nbln = (int)strtol(ep, &ep, 10);
		/* Restore the last byte of the buffer */
		lp->l_line[lp->l_len - 1] = tmp;
		if (nbln < 0)
			errx(1,
			    "invalid line number specification in RCS patch");

		/* find the appropriate line */
		for (;;) {
			if (dlp == NULL)
				break;
			if (dlp->l_lineno == lineno)
				break;
			if (dlp->l_lineno > lineno) {
				dlp = TAILQ_PREV(dlp, tqh, l_list);
			} else if (dlp->l_lineno < lineno) {
				if (((ndlp = TAILQ_NEXT(dlp, l_list)) == NULL) ||
				    ndlp->l_lineno > lineno)
					break;
				dlp = ndlp;
			}
		}
		if (dlp == NULL)
			errx(1, "can't find referenced line in RCS patch");

		if (op == 'd') {
			for (i = 0; (i < nbln) && (dlp != NULL); i++) {
				ndlp = TAILQ_NEXT(dlp, l_list);
				TAILQ_REMOVE(&(dlines->l_lines), dlp, l_list);
				xfree(dlp);
				dlp = ndlp;
				/* last line is gone - reset dlp */
				if (dlp == NULL) {
					ndlp = TAILQ_LAST(&(dlines->l_lines),
					    tqh);
					dlp = ndlp;
				}
			}
		} else if (op == 'a') {
			for (i = 0; i < nbln; i++) {
				ndlp = lp;
				lp = TAILQ_NEXT(lp, l_list);
				if (lp == NULL)
					errx(1, "truncated RCS patch");
				TAILQ_REMOVE(&(plines->l_lines), lp, l_list);
				TAILQ_INSERT_AFTER(&(dlines->l_lines), dlp,
				    lp, l_list);
				dlp = lp;

				/* we don't want lookup to block on those */
				lp->l_lineno = lineno;

				lp = ndlp;
			}
		} else
			errx(1, "unknown RCS patch operation `%c'", op);

		/* last line of the patch, done */
		if (lp->l_lineno == plines->l_nblines)
			break;
	}

	/* once we're done patching, rebuild the line numbers */
	lineno = 0;
	TAILQ_FOREACH(lp, &(dlines->l_lines), l_list)
		lp->l_lineno = lineno++;
	dlines->l_nblines = lineno - 1;

	return (0);
}

/*
 * Split the contents of a file into a list of lines.
 */
static struct rcs_lines *
rcs_splitlines(u_char *data, size_t len)
{
	u_char *c, *p;
	struct rcs_lines *lines;
	struct rcs_line *lp;
	size_t i, tlen;

	lines = xmalloc(sizeof(*lines));
	memset(lines, 0, sizeof(*lines));
	TAILQ_INIT(&(lines->l_lines));

	lp = xmalloc(sizeof(*lp));
	memset(lp, 0, sizeof(*lp));
	TAILQ_INSERT_TAIL(&(lines->l_lines), lp, l_list);


	p = c = data;
	for (i = 0; i < len; i++) {
		if (*p == '\n' || (i == len - 1)) {
			tlen = p - c + 1;
			lp = xmalloc(sizeof(*lp));
			lp->l_line = c;
			lp->l_len = tlen;
			lp->l_lineno = ++(lines->l_nblines);
			TAILQ_INSERT_TAIL(&(lines->l_lines), lp, l_list);
			c = p + 1;
		}
		p++;
	}

	return (lines);
}

static void
rcs_freelines(struct rcs_lines *lines)
{
	struct rcs_line *lp;

	while ((lp = TAILQ_FIRST(&(lines->l_lines))) != NULL) {
		TAILQ_REMOVE(&(lines->l_lines), lp, l_list);
		xfree(lp);
	}

	xfree(lines);
}

static BUF *
rcs_patchfile(u_char *data, size_t dlen, u_char *patch, size_t plen,
    int (*p)(struct rcs_lines *, struct rcs_lines *))
{
	struct rcs_lines *dlines, *plines;
	struct rcs_line *lp;
	BUF *res;

	dlines = rcs_splitlines(data, dlen);
	plines = rcs_splitlines(patch, plen);

	if (p(dlines, plines) < 0) {
		rcs_freelines(dlines);
		rcs_freelines(plines);
		return (NULL);
	}

	res = buf_alloc(1024);
	TAILQ_FOREACH(lp, &dlines->l_lines, l_list) {
		if (lp->l_line == NULL)
			continue;
		buf_append(res, lp->l_line, lp->l_len);
	}

	rcs_freelines(dlines);
	rcs_freelines(plines);
	return (res);
}

/*
 * rcs_expand_keywords()
 *
 * Return expansion any RCS keywords in <data>
 *
 * On error, return NULL.
 */
static BUF *
rcs_expand_keywords(char *rcsfile, struct rcs_delta *rdp, BUF *bp, int mode)
{
	BUF *newbuf;
	u_char *c, *kw, *fin;
	char buf[256];
	u_char *line, *line2;
	u_int i, j;
	int kwtype;
	int found;

	newbuf = buf_alloc(buf_len(bp));

	/*
	 * Keyword formats:
	 * $Keyword$
	 * $Keyword: value$
	 */
	c = buf_get(bp);
	fin = c + buf_len(bp);
	/* Copying to newbuf is deferred until the first keyword. */
	found = 0;

	while (c < fin) {
		kw = memchr(c, '$', fin - c);
		if (kw == NULL)
			break;
		++kw;
		if (found) {
			/* Copy everything up to and including the $ */
			buf_append(newbuf, c, kw - c);
		}
		c = kw;
		/* c points after the $ now */
		if (c == fin)
			break;
		if (!isalpha(*c)) /* All valid keywords start with a letter */
			continue;

		for (i = 0; i < RCS_NKWORDS; ++i) {
			size_t kwlen;

			kwlen = strlen(rcs_expkw[i].kw_str);
			/*
			 * kwlen must be less than clen since clen
			 * includes either a terminating `$' or a `:'.
			 */
			if (c + kwlen < fin &&
			    memcmp(c , rcs_expkw[i].kw_str, kwlen) == 0 &&
			    (c[kwlen] == '$' || c[kwlen] == ':')) {
				c += kwlen;
				break;
			}
		}
		if (i == RCS_NKWORDS)
			continue;
		kwtype = rcs_expkw[i].kw_type;

		/*
		 * if the next character is ':' we need to look for
		 * an '$' before the end of the line to be sure it is
		 * in fact a keyword.
		 */
		if (*c == ':') {
			for (; c < fin; ++c) {
				if (*c == '$' || *c == '\n')
					break;
			}

			if (*c != '$') {
				if (found)
					buf_append(newbuf, kw, c - kw);
				continue;
			}
		}
		++c;

		if (!found) {
			found = 1;
			/* Copy everything up to and including the $ */
			buf_append(newbuf, buf_get(bp), kw - buf_get(bp));
		}

		if (mode & RCS_KWEXP_NAME) {
			buf_puts(newbuf, rcs_expkw[i].kw_str);
			if (mode & RCS_KWEXP_VAL)
				buf_puts(newbuf, ": ");
		}

		/* Order matters because of RCS_KW_ID and RCS_KW_HEADER. */
		if (mode & RCS_KWEXP_VAL) {
			if (kwtype & RCS_KW_RCSFILE) {
				char *tmpf;
				if ((kwtype & RCS_KW_FULLPATH) ||
				    (tmpf = strrchr(rcsfile, '/')) == NULL)
					buf_puts(newbuf, rcsfile);
				else
					buf_puts(newbuf, tmpf + 1);
				buf_putc(newbuf, ' ');
			}

			if (kwtype & RCS_KW_REVISION) {
				rcsnum_tostr(rdp->rd_num, buf, sizeof(buf));
				buf_puts(newbuf, buf);
				buf_putc(newbuf, ' ');
			}

			if (kwtype & RCS_KW_DATE) {
				strftime(buf, sizeof(buf),
				    "%Y/%m/%d %H:%M:%S ", &rdp->rd_date);
				buf_puts(newbuf, buf);
			}

			if (kwtype & RCS_KW_AUTHOR) {
				buf_puts(newbuf, rdp->rd_author);
				buf_putc(newbuf, ' ');
			}

			if (kwtype & RCS_KW_STATE) {
				buf_puts(newbuf, rdp->rd_state);
				buf_putc(newbuf, ' ');
			}

			/* order does not matter anymore below */
			if (kwtype & RCS_KW_SOURCE) {
				buf_puts(newbuf, rcsfile);
				buf_putc(newbuf, ' ');
			}

			if (kwtype & RCS_KW_NAME)
				buf_putc(newbuf, ' ');

			if ((kwtype & RCS_KW_LOCKER)) {
				if (rdp->rd_locker)
					buf_puts(newbuf, rdp->rd_locker);
				buf_putc(newbuf, ' ');
			}
		}

		/* end the expansion */
		if (mode & RCS_KWEXP_NAME)
			buf_putc(newbuf, '$');


		if (kwtype & RCS_KW_LOG) {
			line = memrchr(buf_get(bp), '\n', kw - buf_get(bp) - 1);
			if (line == NULL)
				line = buf_get(bp);
			else
				++line;
			line2 = kw - 1;
			while (line2 > line && line2[-1] == ' ')
				--line2;

			buf_putc(newbuf, '\n');
			buf_append(newbuf, line, kw - 1 - line);
			buf_puts(newbuf, "Revision ");
			rcsnum_tostr(rdp->rd_num, buf, sizeof(buf));
			buf_puts(newbuf, buf);

			buf_puts(newbuf, "  ");
			strftime(buf, sizeof(buf), "%Y/%m/%d %H:%M:%S",
			    &rdp->rd_date);
			buf_puts(newbuf, buf);

			buf_puts(newbuf, "  ");
			buf_puts(newbuf, rdp->rd_author);
			buf_putc(newbuf, '\n');

			for (i = 0; rdp->rd_log[i]; i += j) {
				j = strcspn(rdp->rd_log + i, "\n");
				if (j == 0)
					buf_append(newbuf, line, line2 - line);
				else
					buf_append(newbuf, line, kw - 1 - line);
				if (rdp->rd_log[i + j])
					++j;
				buf_append(newbuf, rdp->rd_log + i, j);
			}
			buf_append(newbuf, line, line2 - line);
			for (j = 0; c + j < fin; ++j) {
				if (c[j] != ' ')
					break;
			}
			if (c[j] == '\n' || c + j == fin)
				c += j;
		}
	}

	if (found) {
		buf_append(newbuf, c, fin - c);
		buf_free(bp);
		return (newbuf);
	} else {
		buf_free(newbuf);
		return (bp);
	}
}

/*
 * rcs_kwexp_buf()
 *
 * Do keyword expansion on a buffer if necessary
 *
 */
static BUF *
rcs_kwexp_buf(BUF *bp, RCSFILE *rf, RCSNUM *rev)
{
	struct rcs_delta *rdp;
	int expmode;

	/*
	 * Do keyword expansion if required.
	 */
	if (rf->rf_expand != NULL)
		expmode = rcs_kwexp_get(rf);
	else
		expmode = RCS_KWEXP_DEFAULT;

	if ((expmode & (RCS_KWEXP_NONE | RCS_KWEXP_OLD)) == 0) {
		if ((rdp = rcs_findrev(rf, rev)) == NULL)
			errx(1, "could not fetch revision");
		return (rcs_expand_keywords(rf->rf_path, rdp, bp, expmode));
	}
	return (bp);
}

static void
rcs_getrevisions_iter(RCSFILE *rfp, struct rcs_delta *rdp, RCSNUM *last,
    BUF *rbuf, BUF *lbuf, rcs_revisions_cb cb, void *cookie)
{
	struct rcs_delta *brdp;
	struct rcs_branch *rb;
	char buf[128];
	char buf2[128];
	int forward;
	BUF *tbuf, *tbuf2;
	size_t dlen;
	u_char *data;

next:
	if (rbuf == NULL) {
		if (rcsnum_cmp(rfp->rf_head, rdp->rd_num, 0) != 0)
			errx(1, "Must be head revision");
		rbuf = buf_alloc(rdp->rd_tlen);
		buf_append(rbuf, rdp->rd_text, rdp->rd_tlen);
		lbuf = buf_alloc(1);
	} else {
		dlen = buf_len(rbuf);
		data = buf_release(rbuf);

		rbuf = rcs_patchfile(data, dlen, rdp->rd_text, rdp->rd_tlen,
		    rcs_patch_lines);
		if (rbuf == NULL)
			errx(1, "Patching failed");
		if (data)
			xfree(data);
	}

	rcsnum_tostr(rdp->rd_num, buf, sizeof(buf));
	if (last) {
		rcsnum_tostr(last, buf2, sizeof(buf2));
		forward = rcsnum_cmp(last, rdp->rd_num, 0) > 0;
	} else {
		forward = 0;
	}

	if (strcmp(rdp->rd_state, "dead") == 0) {
		(*cb)(NULL, 0, NULL, 0, buf, last ? buf2 : NULL, forward,
		    rdp->rd_author, &rdp->rd_date, rdp->rd_log, cookie);
		buf_free(lbuf);
		lbuf = buf_alloc(1);
	} else {
		if (buf_len(rbuf) == 0)
			tbuf = buf_alloc(1);
		else
			tbuf = buf_alloc(buf_len(rbuf));
		buf_append(tbuf, buf_get(rbuf), buf_len(rbuf));
		tbuf = rcs_kwexp_buf(tbuf, rfp, rdp->rd_num);
		(*cb)(buf_get(tbuf), buf_len(tbuf), buf_get(lbuf), buf_len(lbuf),
		    buf, last ? buf2 : NULL,
		    forward, rdp->rd_author, &rdp->rd_date, rdp->rd_log,
		    cookie);
		buf_free(lbuf);
		lbuf = tbuf;
	}

	TAILQ_FOREACH(rb, &rdp->rd_branches, rb_list) {
		brdp = rcs_findrev(rfp, rb->rb_num);
		if (brdp == NULL)
			errx(1, "Branch revision missing");
		if (buf_len(rbuf) == 0)
			tbuf = buf_alloc(1);
		else
			tbuf = buf_alloc(buf_len(rbuf));
		buf_append(tbuf, buf_get(rbuf), buf_len(rbuf));
		if (buf_len(lbuf) == 0) {
			tbuf2 = buf_alloc(1);
		} else {
			tbuf2 = buf_alloc(buf_len(lbuf));
			buf_append(tbuf2, buf_get(lbuf), buf_len(lbuf));
		}
		rcs_getrevisions_iter(rfp, brdp, rdp->rd_num, tbuf, tbuf2, cb, cookie);
	}
	last = rdp->rd_num;

	if (rdp->rd_next->rn_len == 0) {
		buf_free(rbuf);
		buf_free(lbuf);
		return;
	}

	rdp = rcs_findrev(rfp, rdp->rd_next);
	if (rdp == NULL)
		errx(1, "Revision missing");
	goto next;
}

int
rcs_getrevisions(const char *path, rcs_open_cb cb_open, rcs_symbols_cb cb_sym,
    rcs_revisions_cb cb_rev, void *cookie)
{
	RCSFILE *rfp;
	struct rcs_sym *symbol;
	struct rcs_delta *rdp;
	char buf[128];
	int fd;

	fd = open(path, O_RDONLY);
	if (fd == -1) {
		warn("failed to open %s", path);
		return -1;
	}

	rfp = rcs_open(path, fd);
	if (rfp == NULL) {
		close(fd);
		return -1;
	}	

	rdp = rcs_findrev(rfp, rfp->rf_head);
	if (rdp == NULL) {
		warnx("failed to get RCS HEAD revision");
		rcs_close(rfp);
		return -1;
	}

	if (rfp->rf_branch) {
		rcsnum_tostr(rfp->rf_branch, buf, sizeof(buf));
		(*cb_open)(buf, cookie);
	} else {
		(*cb_open)(NULL, cookie);
	}

	TAILQ_FOREACH(symbol, &rfp->rf_symbols, rs_list) {
		rcsnum_tostr(symbol->rs_num, buf, sizeof(buf));
		(*cb_sym)(symbol->rs_name, buf, cookie);
	}

	rcs_getrevisions_iter(rfp, rdp, NULL, NULL, NULL, cb_rev, cookie);

	rcs_close(rfp);
	return 0;
}

static void	 rcsnum_setsize(RCSNUM *, u_int);
static char	*rcsnum_itoa(u_int16_t, char *, size_t);

/*
 * rcsnum_alloc()
 *
 * Allocate an RCS number structure and return a pointer to it.
 */
RCSNUM *
rcsnum_alloc(void)
{
	RCSNUM *rnp;

	rnp = xmalloc(sizeof(*rnp));
	rnp->rn_len = 0;
	rnp->rn_id = NULL;

	return (rnp);
}

/*
 * rcsnum_parse()
 *
 * Parse a string specifying an RCS number and return the corresponding RCSNUM.
 */
static RCSNUM *
rcsnum_parse(const char *str)
{
	char *ep;
	RCSNUM *num;

	num = rcsnum_alloc();
	if (rcsnum_aton(str, &ep, num) < 0 || *ep != '\0') {
		rcsnum_free(num);
		num = NULL;
		if (*ep != '\0')
			warnx("Invalid RCS revision %s", str);
	}

	return (num);
}

/*
 * rcsnum_free()
 *
 * Free an RCSNUM structure previously allocated with rcsnum_alloc().
 */
static void
rcsnum_free(RCSNUM *rn)
{
	if (rn->rn_id != NULL)
		xfree(rn->rn_id);
	xfree(rn);
}

/*
 * rcsnum_tostr()
 *
 * Format the RCS number <nump> into a human-readable dot-separated
 * representation and store the resulting string in <buf>, which is of size
 * <blen>.
 * Returns a pointer to the start of <buf>.  On failure <buf> is set to
 * an empty string.
 */
static char *
rcsnum_tostr(const RCSNUM *nump, char *buf, size_t blen)
{
	u_int i;
	char tmp[8];

	if (nump == NULL || nump->rn_len == 0) {
		buf[0] = '\0';
		return (buf);
	}

	if (strlcpy(buf, rcsnum_itoa(nump->rn_id[0], buf, blen), blen) >= blen)
		errx(1, "rcsnum_tostr: string truncated");
	for (i = 1; i < nump->rn_len; i++) {
		const char *str;

		str = rcsnum_itoa(nump->rn_id[i], tmp, sizeof(tmp));
		if (strlcat(buf, ".", blen) >= blen ||
		    strlcat(buf, str, blen) >= blen)
			errx(1, "rcsnum_tostr: string truncated");
	}

	return (buf);
}

static char *
rcsnum_itoa(u_int16_t num, char *buf, size_t len)
{
	u_int16_t i;
	char *p;

	if (num == 0)
		return strcpy(buf, "0");

	p = buf + len - 1;
	i = num;
	memset(buf, 0, len);
	while (i) {
		*--p = '0' + (i % 10);
		i  /= 10;
	}
	return (p);
}

/*
 * rcsnum_cpy()
 *
 * Copy the number stored in <nsrc> in the destination <ndst> up to <depth>
 * numbers deep.  If <depth> is 0, there is no depth limit.
 */
static void
rcsnum_cpy(const RCSNUM *nsrc, RCSNUM *ndst, u_int depth)
{
	u_int len;

	len = nsrc->rn_len;
	if (depth != 0 && len > depth)
		len = depth;

	rcsnum_setsize(ndst, len);
	/* Overflow checked in rcsnum_setsize(). */
	(void)memcpy(ndst->rn_id, nsrc->rn_id,
	    len * sizeof(*(nsrc->rn_id)));
}

/*
 * rcsnum_cmp()
 *
 * Compare the two numbers <n1> and <n2>. Returns -1 if <n1> is larger than
 * <n2>, 0 if they are both the same, and 1 if <n2> is larger than <n1>.
 * The <depth> argument specifies how many numbers deep should be checked for
 * the result.  A value of 0 means that the depth will be the maximum of the
 * two numbers, so that a longer number is considered greater than a shorter
 * number if they are equal up to the minimum length.
 */
static int
rcsnum_cmp(const RCSNUM *n1, const RCSNUM *n2, u_int depth)
{
	int res;
	u_int i;
	size_t slen;

	slen = MIN(n1->rn_len, n2->rn_len);
	if (depth != 0 && slen > depth)
		slen = depth;

	for (i = 0; i < slen; i++) {
		res = n1->rn_id[i] - n2->rn_id[i];
		if (res < 0)
			return (1);
		else if (res > 0)
			return (-1);
	}

	/* If an explicit depth was specified, and we've
	 * already checked up to depth, consider the
	 * revision numbers equal. */
	if (depth != 0 && slen == depth)
		return (0);
	else if (n1->rn_len > n2->rn_len)
		return (-1);
	else if (n2->rn_len > n1->rn_len)
		return (1);

	return (0);
}

/*
 * rcsnum_aton()
 *
 * Translate the string <str> containing a sequence of digits and periods into
 * its binary representation, which is stored in <nump>.  The address of the
 * first byte not part of the number is stored in <ep> on return, if it is not
 * NULL.
 * Returns 0 on success, or -1 on failure.
 */
static int
rcsnum_aton(const char *str, char **ep, RCSNUM *nump)
{
	u_int32_t val;
	const char *sp;
	char *s;

	if (nump->rn_id == NULL)
		nump->rn_id = xmalloc(sizeof(*(nump->rn_id)));

	nump->rn_len = 0;
	nump->rn_id[0] = 0;

	for (sp = str;; sp++) {
		if (!isdigit((unsigned char)*sp) && (*sp != '.'))
			break;

		if (*sp == '.') {
			if (nump->rn_len >= RCSNUM_MAXLEN - 1) {
				warnx("RCS revision too long: %s", str);
				goto rcsnum_aton_failed;
			}

			nump->rn_len++;
			nump->rn_id = xrealloc(nump->rn_id,
			    nump->rn_len + 1, sizeof(*(nump->rn_id)));
			nump->rn_id[nump->rn_len] = 0;
			continue;
		}

		val = (nump->rn_id[nump->rn_len] * 10) + (*sp - '0');
		if (val > RCSNUM_MAXNUM)
			errx(1, "RCSNUM overflow!");

		nump->rn_id[nump->rn_len] = val;
	}

	if (ep != NULL)
		*(const char **)ep = sp;

	/*
	 * Handle "magic" RCS branch numbers.
	 *
	 * What are they?
	 *
	 * Magic branch numbers have an extra .0. at the second farmost
	 * rightside of the branch number, so instead of having an odd
	 * number of dot-separated decimals, it will have an even number.
	 *
	 * Now, according to all the documentation I've found on the net
	 * about this, cvs does this for "efficiency reasons", I'd like
	 * to hear one.
	 *
	 * We just make sure we remove the .0. from in the branch number.
	 *
	 * XXX - for compatibility reasons with GNU cvs we _need_
	 * to skip this part for the 'log' command, apparently it does
	 * show the magic branches for an unknown and probably
	 * completely insane and not understandable reason in that output.
	 *
	 */
	if (nump->rn_len > 2 && nump->rn_id[nump->rn_len - 1] == 0) {
		/*
		 * Look for ".0.x" at the end of the branch number.
		 */
		if ((s = strrchr(str, '.')) != NULL) {
			s--;
			while (*s != '.')
				s--;

			/*
			 * If we have a "magic" branch, adjust it
			 * so the .0. is removed.
			 */
			if (!strncmp(s, RCS_MAGIC_BRANCH,
			    strlen(RCS_MAGIC_BRANCH))) {
				nump->rn_id[nump->rn_len - 1] =
				    nump->rn_id[nump->rn_len];
				nump->rn_len--;
			}
		}
	}

	/* We can't have a single-digit rcs number. */
	if (nump->rn_len == 0) {
		nump->rn_len++;
		nump->rn_id = xrealloc(nump->rn_id,
		    nump->rn_len + 1, sizeof(*(nump->rn_id)));
		nump->rn_id[nump->rn_len] = 0;
	}

	nump->rn_len++;
	return (nump->rn_len);

rcsnum_aton_failed:
	nump->rn_len = 0;
	xfree(nump->rn_id);
	nump->rn_id = NULL;
	return (-1);
}

/*
 * rcsnum_brtorev()
 *
 * Retrieve the initial revision number associated with the branch number <num>.
 * If <num> is a revision number, an error will be returned.
 */
static RCSNUM *
rcsnum_brtorev(const RCSNUM *brnum)
{
	RCSNUM *num;

	if (!RCSNUM_ISBRANCH(brnum)) {
		return (NULL);
	}

	num = rcsnum_alloc();
	rcsnum_setsize(num, brnum->rn_len + 1);
	rcsnum_cpy(brnum, num, brnum->rn_len);
	num->rn_id[num->rn_len++] = 1;

	return (num);
}

static void
rcsnum_setsize(RCSNUM *num, u_int len)
{
	num->rn_id = xrealloc(num->rn_id, len, sizeof(*(num->rn_id)));
	num->rn_len = len;
}

#define RCS_BUFSIZE	16384
#define RCS_BUFEXTSIZE	8192

/* RCS token types */
#define RCS_TOK_HEAD		(1 << 0)
#define RCS_TOK_BRANCH		(1 << 1)
#define RCS_TOK_ACCESS		(1 << 2)
#define RCS_TOK_SYMBOLS		(1 << 3)
#define RCS_TOK_LOCKS		(1 << 4)
#define RCS_TOK_STRICT		(1 << 5)
#define RCS_TOK_COMMENT		(1 << 6)
#define RCS_TOK_COMMITID	(1 << 7)
#define RCS_TOK_EXPAND		(1 << 8)
#define RCS_TOK_DESC		(1 << 9)
#define RCS_TOK_DATE		(1 << 10)
#define RCS_TOK_AUTHOR		(1 << 11)
#define RCS_TOK_STATE		(1 << 12)
#define RCS_TOK_BRANCHES	(1 << 13)
#define RCS_TOK_NEXT		(1 << 14)
#define RCS_TOK_LOG		(1 << 15)
#define RCS_TOK_TEXT		(1 << 16)
#define RCS_TOK_COLON		(1 << 17)
#define RCS_TOK_COMMA		(1 << 18)
#define RCS_TOK_SCOLON		(1 << 19)

#define RCS_TYPE_STRING		(1 << 20)
#define RCS_TYPE_NUMBER		(1 << 21)
#define RCS_TYPE_BRANCH		(1 << 22)
#define RCS_TYPE_REVISION	(1 << 23)
#define RCS_TYPE_LOGIN		(1 << 24)
#define RCS_TYPE_STATE		(1 << 25)
#define RCS_TYPE_SYMBOL		(1 << 26)
#define RCS_TYPE_DATE		(1 << 27)
#define RCS_TYPE_KEYWORD	(1 << 28)
#define RCS_TYPE_COMMITID	(1 << 29)

#define MANDATORY	0
#define OPTIONAL	1

/* opaque parse data */
struct rcs_pdata {
	char			*rp_buf;
	size_t			 rp_blen;
	char			*rp_bufend;
	size_t			 rp_tlen;

	struct rcs_delta	*rp_delta;
	int			 rp_lineno;
	int			 rp_msglineno;
	int			 rp_token;

	union {
		RCSNUM		*rev;
		char		*str;
		struct tm	 date;
	} rp_value;
};

struct rcs_keyword {
	const char	*k_name;
	int		 k_val;
};

struct rcs_section {
	int	token;
	int	(*parse)(RCSFILE *, struct rcs_pdata *);
	int	opt;
};

/* this has to be sorted always */
static const struct rcs_keyword keywords[] = {
	{ "access",		RCS_TOK_ACCESS},
	{ "author",		RCS_TOK_AUTHOR},
	{ "branch",		RCS_TOK_BRANCH},
	{ "branches",		RCS_TOK_BRANCHES},
	{ "comment",		RCS_TOK_COMMENT},
	{ "commitid",		RCS_TOK_COMMITID},
	{ "date",		RCS_TOK_DATE},
	{ "desc",		RCS_TOK_DESC},
	{ "expand",		RCS_TOK_EXPAND},
	{ "head",		RCS_TOK_HEAD},
	{ "locks",		RCS_TOK_LOCKS},
	{ "log",		RCS_TOK_LOG},
	{ "next",		RCS_TOK_NEXT},
	{ "state",		RCS_TOK_STATE},
	{ "strict",		RCS_TOK_STRICT},
	{ "symbols",		RCS_TOK_SYMBOLS},
	{ "text",		RCS_TOK_TEXT}
};

/* parser functions specified in rcs_section structs */
static int	rcsparse_head(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_branch(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_access(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_symbols(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_locks(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_strict(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_comment(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_commitid(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_expand(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_deltarevision(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_date(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_author(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_state(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_branches(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_next(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_textrevision(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_log(RCSFILE *, struct rcs_pdata *);
static int	rcsparse_text(RCSFILE *, struct rcs_pdata *);

static int	rcsparse_delta(RCSFILE *);
static int	rcsparse_deltatext(RCSFILE *);
static int	rcsparse_desc(RCSFILE *);

static int	kw_cmp(const void *, const void *);
static int	rcsparse(RCSFILE *, const struct rcs_section *);
static void	rcsparse_growbuf(RCSFILE *);
static int	rcsparse_string(RCSFILE *, int);
static int	rcsparse_token(RCSFILE *, int);
static void	rcsparse_warnx(RCSFILE *, const char *, ...);
static int	valid_login(const char *);

/*
 * head [REVISION];
 * [branch BRANCH];
 * access [LOGIN ...];
 * symbols [SYMBOL:REVISION ...];
 * locks [LOGIN:REVISION ...];
 * [strict;]
 * [comment [@[...]@];]
 * [expand [@[...]@];]
 */
static const struct rcs_section sec_admin[] = {
	{ RCS_TOK_HEAD, rcsparse_head, MANDATORY },
	{ RCS_TOK_BRANCH, rcsparse_branch, OPTIONAL },
	{ RCS_TOK_ACCESS, rcsparse_access, MANDATORY },
	{ RCS_TOK_SYMBOLS, rcsparse_symbols, MANDATORY },
	{ RCS_TOK_LOCKS, rcsparse_locks, MANDATORY },
	{ RCS_TOK_STRICT, rcsparse_strict, OPTIONAL },
	{ RCS_TOK_COMMENT, rcsparse_comment, OPTIONAL },
	{ RCS_TOK_EXPAND, rcsparse_expand, OPTIONAL },
	{ 0, NULL, 0 }
};

/*
 * REVISION
 * date [YY]YY.MM.DD.HH.MM.SS;
 * author LOGIN;
 * state STATE;
 * branches [REVISION ...];
 * next [REVISION];
 * [commitid ID;]
 */
static const struct rcs_section sec_delta[] = {
	{ RCS_TYPE_REVISION, rcsparse_deltarevision, MANDATORY },
	{ RCS_TOK_DATE, rcsparse_date, MANDATORY },
	{ RCS_TOK_AUTHOR, rcsparse_author, MANDATORY },
	{ RCS_TOK_STATE, rcsparse_state, MANDATORY },
	{ RCS_TOK_BRANCHES, rcsparse_branches, MANDATORY },
	{ RCS_TOK_NEXT, rcsparse_next, MANDATORY },
	{ RCS_TOK_COMMITID, rcsparse_commitid, OPTIONAL },
	{ 0, NULL, 0 }
};

/*
 * REVISION
 * log @[...]@
 * text @[...]@
 */
static const struct rcs_section sec_deltatext[] = {
	{ RCS_TYPE_REVISION, rcsparse_textrevision, MANDATORY },
	{ RCS_TOK_LOG, rcsparse_log, MANDATORY },
	{ RCS_TOK_TEXT, rcsparse_text, MANDATORY },
	{ 0, NULL, 0 }
};

/*
 * rcsparse_init()
 *
 * Initializes the parsing data structure and parses the admin section of
 * RCS file <rfp>.
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_init(RCSFILE *rfp)
{
	struct rcs_pdata *pdp;

	if (rfp->rf_flags & RCS_PARSED)
		return (0);

	pdp = xmalloc(sizeof(*pdp));
	pdp->rp_buf = xmalloc(RCS_BUFSIZE);
	pdp->rp_blen = RCS_BUFSIZE;
	pdp->rp_bufend = pdp->rp_buf + pdp->rp_blen - 1;
	pdp->rp_token = -1;
	pdp->rp_lineno = 1;
	pdp->rp_msglineno = 1;

	/* ditch the strict lock */
	rfp->rf_flags &= ~RCS_SLOCK;
	rfp->rf_pdata = pdp;

	if (rcsparse(rfp, sec_admin)) {
		rcsparse_free(rfp);
		return (1);
	}

	if (rcsparse_deltatexts(rfp, NULL)) {
		rcsparse_free(rfp);
		return (1);
	}

	rfp->rf_flags |= RCS_SYNCED;
	return (0);
}

/*
 * rcsparse_deltas()
 *
 * Parse deltas. If <rev> is not NULL, parse only as far as that
 * revision. If <rev> is NULL, parse all deltas.
 *
 * Returns 0 on success or 1 on error.
 */
static int
rcsparse_deltas(RCSFILE *rfp, RCSNUM *rev)
{
	int ret;
	struct rcs_delta *enddelta;

	if (rfp->rf_flags & PARSED_DELTAS)
		return (0);

	for (;;) {
		ret = rcsparse_delta(rfp);
		if (rev != NULL) {
			enddelta = TAILQ_LAST(&(rfp->rf_delta), rcs_dlist);
			if (enddelta == NULL)
				return (1);

			if (rcsnum_cmp(enddelta->rd_num, rev, 0) == 0)
				break;
		}

		if (ret == 0) {
			rfp->rf_flags |= PARSED_DELTAS;
			break;
		}
		else if (ret == -1)
			return (1);
	}

	return (0);
}

/*
 * rcsparse_deltatexts()
 *
 * Parse deltatexts. If <rev> is not NULL, parse only as far as that
 * revision. If <rev> is NULL, parse everything.
 *
 * Returns 0 on success or 1 on error.
 */
static int
rcsparse_deltatexts(RCSFILE *rfp, RCSNUM *rev)
{
	int ret;
	struct rcs_delta *rdp;

	if (rfp->rf_flags & PARSED_DELTATEXTS)
		return (0);

	if (!(rfp->rf_flags & PARSED_DESC))
		if (rcsparse_desc(rfp))
			return (1);

	rdp = (rev != NULL) ? rcs_findrev(rfp, rev) : NULL;

	for (;;) {
		if (rdp != NULL && rdp->rd_text != NULL)
			break;
		ret = rcsparse_deltatext(rfp);
		if (ret == 0) {
			rfp->rf_flags |= PARSED_DELTATEXTS;
			break;
		}
		else if (ret == -1)
			return (1);
	}

	return (0);
}

/*
 * rcsparse_free()
 *
 * Free the contents of the <rfp>'s parser data structure.
 */
static void
rcsparse_free(RCSFILE *rfp)
{
	struct rcs_pdata *pdp;

	pdp = rfp->rf_pdata;

	if (pdp->rp_buf != NULL)
		xfree(pdp->rp_buf);
	if (pdp->rp_token == RCS_TYPE_REVISION)
		rcsnum_free(pdp->rp_value.rev);
	xfree(pdp);
}

/*
 * rcsparse_desc()
 *
 * Parse desc of the RCS file <rfp>.  By calling rcsparse_desc, all deltas
 * will be parsed in order to proceed the reading cursor to the desc keyword.
 *
 * desc @[...]@;
 *
 * Returns 0 on success or 1 on error.
 */
static int
rcsparse_desc(RCSFILE *rfp)
{
	struct rcs_pdata *pdp;

	if (rfp->rf_flags & PARSED_DESC)
		return (0);

	if (!(rfp->rf_flags & PARSED_DELTAS) && rcsparse_deltas(rfp, NULL))
		return (1);

	pdp = (struct rcs_pdata *)rfp->rf_pdata;

	if (rcsparse_token(rfp, RCS_TOK_DESC) != RCS_TOK_DESC ||
	    rcsparse_token(rfp, RCS_TYPE_STRING) != RCS_TYPE_STRING)
		return (1);

	rfp->rf_desc = pdp->rp_value.str;
	rfp->rf_flags |= PARSED_DESC;

	return (0);
}

/*
 * rcsparse_deltarevision()
 *
 * Called upon reaching a new REVISION entry in the delta section.
 * A new rcs_delta structure will be prepared in pdp->rp_delta for further
 * parsing.
 *
 * REVISION
 *
 * Always returns 0.
 */
static int
rcsparse_deltarevision(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	struct rcs_delta *rdp;

	rdp = xcalloc(1, sizeof(*rdp));
	TAILQ_INIT(&rdp->rd_branches);
	rdp->rd_num = pdp->rp_value.rev;
	pdp->rp_delta = rdp;

	return (0);
}

/*
 * rcsparse_date()
 *
 * Parses the specified date of current delta pdp->rp_delta.
 *
 * date YYYY.MM.DD.HH.MM.SS;
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_date(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	if (rcsparse_token(rfp, RCS_TYPE_DATE) != RCS_TYPE_DATE)
		return (1);

	pdp->rp_delta->rd_date = pdp->rp_value.date;

	return (rcsparse_token(rfp, RCS_TOK_SCOLON) != RCS_TOK_SCOLON);
}

/*
 * rcsparse_author()
 *
 * Parses the specified author of current delta pdp->rp_delta.
 *
 * author LOGIN;
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_author(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	if (rcsparse_token(rfp, RCS_TYPE_LOGIN) != RCS_TYPE_LOGIN)
		return (1);

	pdp->rp_delta->rd_author = pdp->rp_value.str;

	return (rcsparse_token(rfp, RCS_TOK_SCOLON) != RCS_TOK_SCOLON);
}

/*
 * rcsparse_state()
 *
 * Parses the specified state of current delta pdp->rp_delta.
 *
 * state STATE;
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_state(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	if (rcsparse_token(rfp, RCS_TYPE_STATE) != RCS_TYPE_STATE)
		return (1);

	pdp->rp_delta->rd_state = pdp->rp_value.str;

	return (rcsparse_token(rfp, RCS_TOK_SCOLON) != RCS_TOK_SCOLON);
}

/*
 * rcsparse_branches()
 *
 * Parses the specified branches of current delta pdp->rp_delta.
 *
 * branches [REVISION ...];
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_branches(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	struct rcs_branch *rb;
	int type;

	while ((type = rcsparse_token(rfp, RCS_TOK_SCOLON|RCS_TYPE_REVISION))
	    == RCS_TYPE_REVISION) {
		rb = xmalloc(sizeof(*rb));
		rb->rb_num = pdp->rp_value.rev;
		TAILQ_INSERT_TAIL(&(pdp->rp_delta->rd_branches), rb, rb_list);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_next()
 *
 * Parses the specified next revision of current delta pdp->rp_delta.
 *
 * next [REVISION];
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_next(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	int type;

	type = rcsparse_token(rfp, RCS_TYPE_REVISION|RCS_TOK_SCOLON);
	if (type == RCS_TYPE_REVISION) {
		pdp->rp_delta->rd_next = pdp->rp_value.rev;
		type = rcsparse_token(rfp, RCS_TOK_SCOLON);
	} else
		pdp->rp_delta->rd_next = rcsnum_alloc();

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_commitid()
 *
 * Parses the specified commit id of current delta pdp->rp_delta. The
 * commitid keyword is optional and can be omitted.
 *
 * [commitid ID;]
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_commitid(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	if (rcsparse_token(rfp, RCS_TYPE_COMMITID) != RCS_TYPE_COMMITID)
		return (1);

	/* XXX - do something with commitid */

	return (rcsparse_token(rfp, RCS_TOK_SCOLON) != RCS_TOK_SCOLON);
}

/*
 * rcsparse_textrevision()
 *
 * Called upon reaching a new REVISION entry in the delta text section.
 * pdp->rp_delta will be set to REVISION's delta (created in delta section)
 * for further parsing.
 *
 * REVISION
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_textrevision(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	struct rcs_delta *rdp;

	TAILQ_FOREACH(rdp, &rfp->rf_delta, rd_list) {
		if (rcsnum_cmp(rdp->rd_num, pdp->rp_value.rev, 0) == 0)
			break;
	}
	if (rdp == NULL) {
		rcsparse_warnx(rfp, "delta for revision \"%s\" not found",
		    pdp->rp_buf);
		rcsnum_free(pdp->rp_value.rev);
		return (1);
	}
	pdp->rp_delta = rdp;

	rcsnum_free(pdp->rp_value.rev);
	return (0);
}

/*
 * rcsparse_log()
 *
 * Parses the specified log of current deltatext pdp->rp_delta.
 *
 * log @[...]@
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_log(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	if (rcsparse_token(rfp, RCS_TYPE_STRING) != RCS_TYPE_STRING)
		return (1);

	pdp->rp_delta->rd_log = pdp->rp_value.str;

	return (0);
}

/*
 * rcsparse_text()
 *
 * Parses the specified text of current deltatext pdp->rp_delta.
 *
 * text @[...]@
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_text(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	if (rcsparse_token(rfp, RCS_TYPE_STRING) != RCS_TYPE_STRING)
		return (1);

	pdp->rp_delta->rd_tlen = pdp->rp_tlen - 1;
	if (pdp->rp_delta->rd_tlen == 0) {
		pdp->rp_delta->rd_text = (unsigned char *)xstrdup("");
	} else {
		pdp->rp_delta->rd_text = xmalloc(pdp->rp_delta->rd_tlen);
		memcpy(pdp->rp_delta->rd_text, pdp->rp_buf,
		    pdp->rp_delta->rd_tlen);
	}
	xfree(pdp->rp_value.str);

	return (0);
}

/*
 * rcsparse_head()
 *
 * Parses the head revision of RCS file <rfp>.
 *
 * head [REVISION];
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_head(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	int type;

	type = rcsparse_token(rfp, RCS_TYPE_REVISION|RCS_TOK_SCOLON);
	if (type == RCS_TYPE_REVISION) {
		rfp->rf_head = pdp->rp_value.rev;
		type = rcsparse_token(rfp, RCS_TOK_SCOLON);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_branch()
 *
 * Parses the default branch of RCS file <rfp>. The branch keyword is
 * optional and can be omitted.
 *
 * [branch BRANCH;]
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_branch(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	int type;

	type = rcsparse_token(rfp, RCS_TYPE_BRANCH|RCS_TOK_SCOLON);
	if (type == RCS_TYPE_BRANCH) {
		rfp->rf_branch = pdp->rp_value.rev;
		type = rcsparse_token(rfp, RCS_TOK_SCOLON);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_access()
 *
 * Parses the access list of RCS file <rfp>.
 *
 * access [LOGIN ...];
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_access(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	struct rcs_access *ap;
	int type;

	while ((type = rcsparse_token(rfp, RCS_TOK_SCOLON|RCS_TYPE_LOGIN))
	    == RCS_TYPE_LOGIN) {
		ap = xmalloc(sizeof(*ap));
		ap->ra_name = pdp->rp_value.str;
		TAILQ_INSERT_TAIL(&(rfp->rf_access), ap, ra_list);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_symbols()
 *
 * Parses the symbol list of RCS file <rfp>.
 *
 * symbols [SYMBOL:REVISION ...];
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_symbols(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	struct rcs_sym *symp;
	char *name;
	int type;

	while ((type = rcsparse_token(rfp, RCS_TOK_SCOLON|RCS_TYPE_SYMBOL)) ==
	    RCS_TYPE_SYMBOL) {
		name = pdp->rp_value.str;
		if (rcsparse_token(rfp, RCS_TOK_COLON) != RCS_TOK_COLON ||
		    rcsparse_token(rfp, RCS_TYPE_NUMBER) != RCS_TYPE_NUMBER) {
			xfree(name);
			return (1);
		}
		symp = xmalloc(sizeof(*symp));
		symp->rs_name = name;
		symp->rs_num = pdp->rp_value.rev;
		TAILQ_INSERT_TAIL(&(rfp->rf_symbols), symp, rs_list);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_locks()
 *
 * Parses the lock list of RCS file <rfp>.
 *
 * locks [SYMBOL:REVISION ...];
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_locks(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	struct rcs_lock *lkp;
	char *name;
	int type;

	while ((type = rcsparse_token(rfp, RCS_TOK_SCOLON|RCS_TYPE_LOGIN)) ==
	    RCS_TYPE_LOGIN) {
		name = pdp->rp_value.str;
		if (rcsparse_token(rfp, RCS_TOK_COLON) != RCS_TOK_COLON ||
		    rcsparse_token(rfp, RCS_TYPE_REVISION) !=
		    RCS_TYPE_REVISION) {
			xfree(name);
			return (1);
		}
		lkp = xmalloc(sizeof(*lkp));
		lkp->rl_name = name;
		lkp->rl_num = pdp->rp_value.rev;
		TAILQ_INSERT_TAIL(&(rfp->rf_locks), lkp, rl_list);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_locks()
 *
 * Parses the strict keyword of RCS file <rfp>. The strict keyword is
 * optional and can be omitted.
 *
 * [strict;]
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_strict(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	rfp->rf_flags |= RCS_SLOCK;

	return (rcsparse_token(rfp, RCS_TOK_SCOLON) != RCS_TOK_SCOLON);
}

/*
 * rcsparse_comment()
 *
 * Parses the comment of RCS file <rfp>.  The comment keyword is optional
 * and can be omitted.
 *
 * [comment [@[...]@];]
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_comment(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	int type;

	type = rcsparse_token(rfp, RCS_TYPE_STRING|RCS_TOK_SCOLON);
	if (type == RCS_TYPE_STRING) {
		rfp->rf_comment = pdp->rp_value.str;
		type = rcsparse_token(rfp, RCS_TOK_SCOLON);
	}

	return (type != RCS_TOK_SCOLON);
}

/*
 * rcsparse_expand()
 *
 * Parses expand of RCS file <rfp>.  The expand keyword is optional and
 * can be omitted.
 *
 * [expand [@[...]@];]
 *
 * Returns 0 on success or 1 on failure.
 */
static int
rcsparse_expand(RCSFILE *rfp, struct rcs_pdata *pdp)
{
	int type;

	type = rcsparse_token(rfp, RCS_TYPE_STRING|RCS_TOK_SCOLON);
	if (type == RCS_TYPE_STRING) {
		rfp->rf_expand = pdp->rp_value.str;
		type = rcsparse_token(rfp, RCS_TOK_SCOLON);
	}

	return (type != RCS_TOK_SCOLON);
}

#define RBUF_PUTC(ch) \
do { \
	if (bp == pdp->rp_bufend - 1) { \
		len = bp - pdp->rp_buf; \
		rcsparse_growbuf(rfp); \
		bp = pdp->rp_buf + len; \
	} \
	*(bp++) = (ch); \
	pdp->rp_tlen++; \
} while (0);

static int
rcsparse_string(RCSFILE *rfp, int allowed)
{
	struct rcs_pdata *pdp;
	int c;
	size_t len;
	char *bp;

	pdp = (struct rcs_pdata *)rfp->rf_pdata;

	bp = pdp->rp_buf;
	pdp->rp_tlen = 0;
	*bp = '\0';

	for (;;) {
		c = getc(rfp->rf_file);
		if (c == '@') {
			c = getc(rfp->rf_file);
			if (c == EOF) {
				return (EOF);
			} else if (c != '@') {
				ungetc(c, rfp->rf_file);
				break;
			}
		}

		if (c == EOF) {
			return (EOF);
		} else if (c == '\n')
			pdp->rp_lineno++;

		RBUF_PUTC(c);
	}

	bp = pdp->rp_buf + pdp->rp_tlen;
	RBUF_PUTC('\0');

	if (!(allowed & RCS_TYPE_STRING)) {
		rcsparse_warnx(rfp, "unexpected RCS string");
		return (0);
	}

	pdp->rp_value.str = xstrdup(pdp->rp_buf);

	return (RCS_TYPE_STRING);
}

static int
rcsparse_token(RCSFILE *rfp, int allowed)
{
	const struct rcs_keyword *p;
	struct rcs_pdata *pdp;
	int c, pre, ret, type;
	char *bp;
	size_t len;
	RCSNUM *datenum;

	pdp = (struct rcs_pdata *)rfp->rf_pdata;

	if (pdp->rp_token != -1) {
		/* no need to check for allowed here */
		type = pdp->rp_token;
		pdp->rp_token = -1;
		return (type);
	}

	/* skip whitespaces */
	c = EOF;
	do {
		pre = c;
		c = getc(rfp->rf_file);
		if (c == EOF) {
			if (ferror(rfp->rf_file)) {
				rcsparse_warnx(rfp, "error during parsing");
				return (0);
			}
			if (pre != '\n')
				rcsparse_warnx(rfp,
				    "no newline at end of file");
			return (EOF);
		} else if (c == '\n')
			pdp->rp_lineno++;
	} while (isspace(c));

	pdp->rp_msglineno = pdp->rp_lineno;
	type = 0;
	switch (c) {
	case '@':
		ret = rcsparse_string(rfp, allowed);
		if (ret == EOF && ferror(rfp->rf_file)) {
			rcsparse_warnx(rfp, "error during parsing");
			return (0);
		}
		return (ret);
		/* NOTREACHED */
        case ':':
		type = RCS_TOK_COLON;
		if (type & allowed)
			return (type);
		rcsparse_warnx(rfp, "unexpected token \"%c\"", c);
		return (0);
		/* NOTREACHED */
        case ';':
		type = RCS_TOK_SCOLON;
		if (type & allowed)
			return (type);
		rcsparse_warnx(rfp, "unexpected token \"%c\"", c);
		return (0);
		/* NOTREACHED */
        case ',':
		type = RCS_TOK_COMMA;
		if (type & allowed)
			return (type);
		rcsparse_warnx(rfp, "unexpected token \"%c\"", c);
		return (0);
		/* NOTREACHED */
	default:
		if (!isgraph(c)) {
			rcsparse_warnx(rfp, "unexpected character 0x%.2X", c);
			return (0);
		}
		break;
	}
	allowed &= ~(RCS_TOK_COLON|RCS_TOK_SCOLON|RCS_TOK_COMMA);

	bp = pdp->rp_buf;
	pdp->rp_tlen = 0;
	*bp = '\0';

	for (;;) {
		if (c == EOF) {
			if (ferror(rfp->rf_file))
				rcsparse_warnx(rfp, "error during parsing");
			else
				rcsparse_warnx(rfp, "unexpected end of file");
			return (0);
		} else if (c == '\n')
			pdp->rp_lineno++;

		RBUF_PUTC(c);

		c = getc(rfp->rf_file);

		if (isspace(c)) {
			if (c == '\n')
				pdp->rp_lineno++;
			RBUF_PUTC('\0');
			break;
		} else if (c == ';' || c == ':' || c == ',') {
			ungetc(c, rfp->rf_file);
			RBUF_PUTC('\0');
			break;
		} else if (!isgraph(c)) {
			rcsparse_warnx(rfp, "unexpected character 0x%.2X", c);
			return (0);
		}
	}

	switch (allowed) {
	case RCS_TYPE_COMMITID:
		/* XXX validate commitd */
		break;
	case RCS_TYPE_LOGIN:
		if (!valid_login(pdp->rp_buf)) {
			rcsparse_warnx(rfp, "invalid login \"%s\"",
			    pdp->rp_buf);
			return (0);
		}
		pdp->rp_value.str = xstrdup(pdp->rp_buf);
		break;
	case RCS_TYPE_SYMBOL:
		if (!rcs_sym_check(pdp->rp_buf)) {
			rcsparse_warnx(rfp, "invalid symbol \"%s\"",
			    pdp->rp_buf);
			return (0);
		}
		pdp->rp_value.str = xstrdup(pdp->rp_buf);
		break;
		/* FALLTHROUGH */
	case RCS_TYPE_STATE:
		if (rcs_state_check(pdp->rp_buf)) {
			rcsparse_warnx(rfp, "invalid state \"%s\"",
			    pdp->rp_buf);
			return (0);
		}
		pdp->rp_value.str = xstrdup(pdp->rp_buf);
		break;
	case RCS_TYPE_DATE:
		if ((datenum = rcsnum_parse(pdp->rp_buf)) == NULL) {
			rcsparse_warnx(rfp, "invalid date \"%s\"", pdp->rp_buf);
			return (0);
		}
		if (datenum->rn_len != 6) {
			rcsnum_free(datenum);
			rcsparse_warnx(rfp, "invalid date \"%s\"", pdp->rp_buf);
			return (0);
		}
		pdp->rp_value.date.tm_year = datenum->rn_id[0];
		if (pdp->rp_value.date.tm_year >= 1900)
			pdp->rp_value.date.tm_year -= 1900;
		pdp->rp_value.date.tm_mon = datenum->rn_id[1] - 1;
		pdp->rp_value.date.tm_mday = datenum->rn_id[2];
		pdp->rp_value.date.tm_hour = datenum->rn_id[3];
		pdp->rp_value.date.tm_min = datenum->rn_id[4];
		pdp->rp_value.date.tm_sec = datenum->rn_id[5];
		rcsnum_free(datenum);
		break;
	case RCS_TYPE_NUMBER:
		pdp->rp_value.rev = rcsnum_parse(pdp->rp_buf);
		if (pdp->rp_value.rev == NULL) {
			rcsparse_warnx(rfp, "invalid number \"%s\"",
			    pdp->rp_buf);
			return (0);
		}
		break;
	case RCS_TYPE_BRANCH:
		pdp->rp_value.rev = rcsnum_parse(pdp->rp_buf);
		if (pdp->rp_value.rev == NULL) {
			rcsparse_warnx(rfp, "invalid branch \"%s\"",
			    pdp->rp_buf);
			return (0);
		}
		if (!RCSNUM_ISBRANCH(pdp->rp_value.rev)) {
			rcsnum_free(pdp->rp_value.rev);
			rcsparse_warnx(rfp, "expected branch, got \"%s\"",
			    pdp->rp_buf);
			return (0);
		}
		break;
	case RCS_TYPE_KEYWORD:
		if (islower((unsigned char)*pdp->rp_buf)) {
			p = bsearch(pdp->rp_buf, keywords,
			    sizeof(keywords) / sizeof(keywords[0]),
			    sizeof(keywords[0]), kw_cmp);
			if (p != NULL)
				return (p->k_val);
		}
		allowed = RCS_TYPE_REVISION;
		/* FALLTHROUGH */
	case RCS_TYPE_REVISION:
		pdp->rp_value.rev = rcsnum_parse(pdp->rp_buf);
		if (pdp->rp_value.rev != NULL) {
			if (RCSNUM_ISBRANCH(pdp->rp_value.rev)) {
				rcsnum_free(pdp->rp_value.rev);
				rcsparse_warnx(rfp,
				    "expected revision, got \"%s\"",
				    pdp->rp_buf);
				return (0);
			}
			break;
		}
		/* FALLTHROUGH */
	default:
		RBUF_PUTC('\0');
		rcsparse_warnx(rfp, "unexpected token \"%s\"", pdp->rp_buf);
		return (0);
		/* NOTREACHED */
	}

	return (allowed);
}

static int
rcsparse(RCSFILE *rfp, const struct rcs_section *sec)
{
	struct rcs_pdata *pdp;
	int i, token;

	pdp = (struct rcs_pdata *)rfp->rf_pdata;
	i = 0;

	token = 0;
	for (i = 0; sec[i].token != 0; i++) {
		token = rcsparse_token(rfp, RCS_TYPE_KEYWORD);
		if (token == 0)
			return (1);

		while (token != sec[i].token) {
			if (sec[i].parse == NULL)
				goto end;
			if (sec[i].opt) {
				i++;
				continue;
			}
			if (token == EOF || (!(rfp->rf_flags & PARSED_DELTAS) &&
			    token == RCS_TOK_DESC))
				goto end;
			rcsparse_warnx(rfp, "unexpected token \"%s\"",
			    pdp->rp_buf);
			return (1);
		}

		if (sec[i].parse(rfp, pdp))
			return (1);
	}
end:
	if (token == RCS_TYPE_REVISION)
		pdp->rp_token = token;
	else if (token == RCS_TOK_DESC)
		pdp->rp_token = RCS_TOK_DESC;
	else if (token == EOF)
		rfp->rf_flags |= RCS_PARSED;

	return (0);
}

static int
rcsparse_deltatext(RCSFILE *rfp)
{
	struct rcs_pdata *pdp;
	int ret;

	if (rfp->rf_flags & PARSED_DELTATEXTS)
		return (0);

	if (!(rfp->rf_flags & PARSED_DESC))
		if ((ret = rcsparse_desc(rfp)))
			return (ret);
		
	pdp = (struct rcs_pdata *)rfp->rf_pdata;

	if (rcsparse(rfp, sec_deltatext))
		return (-1);

	if (rfp->rf_flags & RCS_PARSED)
		rfp->rf_flags |= PARSED_DELTATEXTS;

	return (1);
}

static int
rcsparse_delta(RCSFILE *rfp)
{
	struct rcs_pdata *pdp;

	if (rfp->rf_flags & PARSED_DELTAS)
		return (0);

	pdp = (struct rcs_pdata *)rfp->rf_pdata;
	if (pdp->rp_token == RCS_TOK_DESC) {
		rfp->rf_flags |= PARSED_DELTAS;
		return (0);
	}

	if (rcsparse(rfp, sec_delta))
		return (-1);

	if (pdp->rp_delta != NULL) {
		TAILQ_INSERT_TAIL(&rfp->rf_delta, pdp->rp_delta, rd_list);
		pdp->rp_delta = NULL;
		rfp->rf_ndelta++;
		return (1);
	}

	return (0);
}

/*
 * rcsparse_growbuf()
 *
 * Attempt to grow the internal parse buffer for the RCS file <rf> by
 * RCS_BUFEXTSIZE.
 * In case of failure, the original buffer is left unmodified.
 */
static void
rcsparse_growbuf(RCSFILE *rfp)
{
	struct rcs_pdata *pdp = (struct rcs_pdata *)rfp->rf_pdata;
	
	pdp->rp_buf = xrealloc(pdp->rp_buf, 1,
		pdp->rp_blen + RCS_BUFEXTSIZE);
	pdp->rp_blen += RCS_BUFEXTSIZE;
	pdp->rp_bufend = pdp->rp_buf + pdp->rp_blen - 1;
}

/*
 * Borrowed from src/usr.sbin/user/user.c:
 * return 1 if `login' is a valid login name
 */
static int
valid_login(const char *login_name)
{
	const unsigned char *cp;

	/* The first character cannot be a hyphen */
	if (*login_name == '-')
		return 0;

	for (cp = (const unsigned char *)login_name ; *cp ; cp++) {
		/* We allow '$' as the last character for samba */
		if (!isalnum(*cp) && *cp != '.' && *cp != '_' && *cp != '-' &&
		    !(*cp == '$' && *(cp + 1) == '\0')) {
			return 0;
		}
	}
	return 1;
}

static int
kw_cmp(const void *k, const void *e)
{
	return (strcmp(k, ((const struct rcs_keyword *)e)->k_name));
}

static void
rcsparse_warnx(RCSFILE *rfp, const char *fmt, ...)
{
	struct rcs_pdata *pdp;
	va_list ap;
	char *nfmt;

	pdp = (struct rcs_pdata *)rfp->rf_pdata;
	va_start(ap, fmt);
	xasprintf(&nfmt, "%s:%d: %s", rfp->rf_path, pdp->rp_msglineno, fmt);
	vwarnx(nfmt, ap);
	va_end(ap);
	free(nfmt);
}
