/*	$OpenBSD: rcs.h,v 1.13 2010/10/20 19:53:53 tobias Exp $	*/
/*
 * Copyright (c) 2004 Jean-Francois Brousseau <jfb@openbsd.org>
 * Copyright (c) 2006 Xavier Santolaria <xsa@openbsd.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL  DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef RCS_H
#define RCS_H

#include <sys/queue.h>

#include <stdio.h>
#include <time.h>

#define RCS_STATE_INVALCHAR	"$,:;@"
#define RCS_SYM_INVALCHAR	"$,.:;@"

#define RCS_MAGIC_BRANCH	".0."

/*
 * Keyword expansion table
 */
#define RCS_KW_AUTHOR		0x1000
#define RCS_KW_DATE		0x2000
#define RCS_KW_LOG		0x4000
#define RCS_KW_NAME		0x8000
#define RCS_KW_RCSFILE		0x0100
#define RCS_KW_REVISION		0x0200
#define RCS_KW_SOURCE		0x0400
#define RCS_KW_STATE		0x0800
#define RCS_KW_FULLPATH		0x0010
#define RCS_KW_LOCKER		0x0020

#define RCS_KW_ID \
	(RCS_KW_RCSFILE | RCS_KW_REVISION | RCS_KW_DATE \
	| RCS_KW_AUTHOR | RCS_KW_STATE)

#define RCS_KW_HEADER	(RCS_KW_ID | RCS_KW_FULLPATH)

/* RCS keyword expansion modes (kflags) */
#define RCS_KWEXP_NONE	0x01	/* do not expand keywords */
#define RCS_KWEXP_NAME	0x02	/* include keyword name */
#define RCS_KWEXP_VAL	0x04	/* include keyword value */
#define RCS_KWEXP_LKR	0x08	/* include name of locker */
#define RCS_KWEXP_OLD	0x10	/* generate old keyword string */
#define RCS_KWEXP_ERR	0x20	/* mode has an error */

#define RCS_KWEXP_DEFAULT	(RCS_KWEXP_NAME | RCS_KWEXP_VAL)
#define RCS_KWEXP_KVL		(RCS_KWEXP_NAME | RCS_KWEXP_VAL | RCS_KWEXP_LKR)

#define RCS_KWEXP_INVAL(k) \
	((k & RCS_KWEXP_ERR) || \
	((k & RCS_KWEXP_OLD) && (k & ~RCS_KWEXP_OLD)))

struct rcs_kw {
	char	kw_str[16];
	int	kw_type;
};

#define RCS_NKWORDS	(sizeof(rcs_expkw)/sizeof(rcs_expkw[0]))

#define RCSNUM_MAXNUM	65535
#define RCSNUM_MAXLEN	64

#define RCSNUM_ISBRANCH(n)	((n)->rn_len % 2)
#define RCSNUM_ISBRANCHREV(n)	(!((n)->rn_len % 2) && ((n)->rn_len >= 4))

/* internal flags */
#define RCS_PARSED	  (1<<4)  /* file has been parsed */
#define RCS_SYNCED	  (1<<5)  /* in-mem copy is sync with disk copy */
#define RCS_SLOCK	  (1<<6)  /* strict lock */

/* parser flags */
#define PARSED_DELTAS     (1<<7)  /* all deltas are parsed */
#define PARSED_DESC       (1<<8)  /* the description is parsed */
#define PARSED_DELTATEXTS (1<<9)  /* all delta texts are parsed */

/* delta flags */
#define RCS_RD_DEAD	0x01	/* dead */
#define RCS_RD_SELECT	0x02	/* select for operation */

/* RCS error codes */
#define RCS_ERR_NOERR	0
#define RCS_ERR_NOENT	1
#define RCS_ERR_DUPENT	2
#define RCS_ERR_BADNUM	3
#define RCS_ERR_BADSYM	4
#define RCS_ERR_PARSE	5
#define RCS_ERR_ERRNO	255

typedef struct rcs_num {
	u_int		 rn_len;
	u_int16_t	*rn_id;
} RCSNUM;

struct rcs_access {
	char			*ra_name;
	TAILQ_ENTRY(rcs_access)	 ra_list;
};

struct rcs_sym {
	char			*rs_name;
	RCSNUM			*rs_num;
	TAILQ_ENTRY(rcs_sym)	 rs_list;
};

struct rcs_lock {
	char	*rl_name;
	RCSNUM	*rl_num;

	TAILQ_ENTRY(rcs_lock)	 rl_list;
};

struct rcs_branch {
	RCSNUM			*rb_num;
	TAILQ_ENTRY(rcs_branch)	 rb_list;
};

TAILQ_HEAD(rcs_dlist, rcs_delta);

struct rcs_delta {
	RCSNUM		*rd_num;
	RCSNUM		*rd_next;
	u_int		 rd_flags;
	struct tm	 rd_date;
	char		*rd_author;
	char		*rd_state;
	char		*rd_log;
	char		*rd_locker;
	u_char		*rd_text;
	size_t		 rd_tlen;

	TAILQ_HEAD(, rcs_branch)	rd_branches;
	TAILQ_ENTRY(rcs_delta)		rd_list;
};


typedef struct rcs_file {
	FILE	*rf_file;
	char	*rf_path;
	mode_t	 rf_mode;
	u_int	 rf_flags;

	RCSNUM	*rf_head;
	RCSNUM	*rf_branch;
	char	*rf_comment;
	char	*rf_expand;
	char	*rf_desc;

	u_int					rf_ndelta;
	struct rcs_dlist			rf_delta;
	TAILQ_HEAD(rcs_alist, rcs_access)	rf_access;
	TAILQ_HEAD(rcs_slist, rcs_sym)		rf_symbols;
	TAILQ_HEAD(rcs_llist, rcs_lock)		rf_locks;

	void	*rf_pdata;
} RCSFILE;

static struct rcs_delta	*rcs_findrev(RCSFILE *, RCSNUM *);

static RCSNUM	*rcsnum_alloc(void);
static RCSNUM	*rcsnum_parse(const char *);
static RCSNUM	*rcsnum_brtorev(const RCSNUM *);
static void	 rcsnum_free(RCSNUM *);
static int	 rcsnum_aton(const char *, char **, RCSNUM *);
static char	*rcsnum_tostr(const RCSNUM *, char *, size_t);
static void	 rcsnum_cpy(const RCSNUM *, RCSNUM *, u_int);
static int	 rcsnum_cmp(const RCSNUM *, const RCSNUM *, u_int);

#include "rcs-public.h"

#endif	/* RCS_H */
