#include <assert.h>
#include <err.h>
#include <sqlite3.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "common.h"

static const char create_statement[] =
    "PRAGMA cache_size=16384;\n"
    "PRAGMA synchronous=off;\n"
    "BEGIN;\n"
    "DROP TABLE IF EXISTS branchpoints2;\n"
    "DROP TABLE IF EXISTS revbranches2;\n"
    "DROP TABLE IF EXISTS skiprev2;\n"
    "CREATE TABLE branchpoints2 ("
	"branch integer NOT NULL REFERENCES branches(id), "
	"revision integer NULL REFERENCES revision(id), "
	"parent integer NULL REFERENCES branches(id), "
	"file integer NOT NULL REFERENCES file(id)"
    ");\n"
    "CREATE TABLE revbranches2 ("
        "branch integer NOT NULL REFERENCES branches(id), "
	"revision integer NOT NULL REFERENCES revision(id) UNIQUE, "
	"file integer NOT NULL REFERENCES file(id)"
    ");\n"
    "CREATE TABLE skiprev2 ("
	"revision integer NULL REFERENCES revision(id), "
	"file integer NOT NULL REFERENCES file(id)"
    ");\n"
    "CREATE TEMPORARY TABLE processed_branches ("
        "branch integer NOT NULL REFERENCES branches(id)"
    ");\n"
    "CREATE INDEX branchpoints2_branch ON branchpoints2(branch);\n"
    "CREATE INDEX branchpoints2_parent ON branchpoints2(parent);\n"
    "INSERT INTO branchpoints2 SELECT * FROM branchpoints;\n"
    "INSERT INTO revbranches2 SELECT * FROM revbranches;\n"
    "INSERT INTO skiprev2 SELECT * FROM skiprev;\n";

static const char skip_initial_revision_statement[] =
    "INSERT INTO skiprev2 SELECT r.id, r.file FROM revision r, revision vr, "
    "revision_link rl, revbranches2 rb, branches b "
    "WHERE r.revision=\"1.1\" AND r.message=\"Initial revision\n\" AND "
    "vr.id=rl.child AND rl.parent=r.id AND rb.revision=rl.child AND "
    "rb.branch=b.id AND b.vendorbranch AND "
    "datetime(vr.date) BETWEEN datetime(r.date, '-10 seconds') AND "
    "datetime(r.date, '+10 seconds') AND "
    "vr.author=r.author AND NOT EXISTS(SELECT * FROM branchpoints bp, "
    "branches b WHERE bp.branch=b.id AND NOT b.vendorbranch AND "
    "bp.revision=r.id);\n"
    "INSERT INTO skiprev2 SELECT r.id, r.file FROM revision r "
    "WHERE r.revision=\"1.1\" AND r.content ISNULL";

static const char defaultbranch_statement[] =
    "UPDATE revbranches2 SET branch=(SELECT id FROM branches "
    "WHERE symbol=\".HEAD\") WHERE "
    "file IN (SELECT id FROM file WHERE branch=\"1.1.1\") AND "
    "branch IN (SELECT id FROM branches WHERE vendorbranch)";

static const char no_defaultbranch_statement[] =
    "UPDATE revbranches2 SET branch=(SELECT id FROM branches "
    "WHERE symbol=\".HEAD\") WHERE "
    "revision IN (SELECT r.id FROM file f, revision r, revision r2, "
    "revbranches2 rb, branches b WHERE b.vendorbranch AND rb.branch=b.id AND "
    "r.id=rb.revision AND r.file=f.id AND f.branch ISNULL AND r2.id IN "
    "(SELECT id FROM revision WHERE (revision=\"1.2\" OR revision=\"1.3\" OR "
    "revision=\"1.4\") AND file=f.id ORDER BY date LIMIT 1) "
    "AND r.date < r2.date)";

static const char brachpoints_statement[] =
    "UPDATE branchpoints2 SET parent=(SELECT id FROM branches "
    "WHERE symbol=\".HEAD\") WHERE revision IN (SELECT r.id FROM revision r, "
    "revbranches2 rb, branches b WHERE rb.branch=b.id AND b.symbol=\".HEAD\" "
    "AND rb.revision=r.id)";

static const char branchpoints_warning_statement[] =
    "SELECT DISTINCT b.symbol, b2.symbol FROM branchpoints2 bp, branches b, "
    "branches b2 WHERE b.id=bp.branch AND b2.id=bp.parent ORDER BY b.symbol";

static const char parentless_branches[] =
    "INSERT INTO processed_branches SELECT b.id FROM branches b "
    "WHERE NOT EXISTS(SELECT * FROM branchpoints2 bp WHERE bp.branch=b.id)";

static const char next_unchecked_branches[] =
    "SELECT b.id FROM branches b WHERE "
    "NOT EXISTS(SELECT * FROM processed_branches pb WHERE pb.branch=b.id) "
    "LIMIT 1";

static const char process_unchecked_branches[] =
    "INSERT INTO processed_branches VALUES (?)\n";

static const char process_unchecked_branches2[] =
    "UPDATE branchpoints2 SET parent=? WHERE branch IN "
    "(SELECT branch FROM branchpoints2 WHERE parent=?)";

static char *last_branch, *last_parent;

static int
branch_warning(void *cookie, int columns, char **values, char **names)
{
	assert(columns == 2);

	if (last_branch && strcmp(last_branch, values[0]) == 0) {
		if (last_parent) {
			printf("Inconsistent branch points for %s, "
			    "candidates:\n", last_branch);
			puts(last_parent);
			last_parent = NULL;
		}
		puts(values[1]);
	} else {
		free(last_branch);
		free(last_parent);
		if ((last_branch = strdup(values[0])) == NULL)
			err(1, "strdup failed");
		if ((last_parent = strdup(values[1])) == NULL)
			err(1, "strdup failed");
	}
	return 0;
}

static void
usage(void)
{
	fprintf(stderr, "Usage: %s database", getprogname());
	exit(1);
}

int
main(int argc, char **argv)
{
	sqlite3 *db;
	sqlite3_stmt *next_stmt, *process_stmt, *process_stmt2;
	sqlite3_int64 id;
	char *errmsg;
	int ch, rv;

	while ((ch = getopt(argc, argv, "")) != -1) {
		switch (ch) {
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 1)
		usage();

	if (sqlite3_open_v2(argv[0], &db, SQLITE_OPEN_READWRITE, NULL))
		errx(1, "database open failed: %s", sqlite3_errmsg(db));

	if (sqlite3_exec(db, create_statement, NULL, 0, &errmsg))
		errx(1, "schema update failed: %s", errmsg);

	if (sqlite3_exec(db, skip_initial_revision_statement, NULL, NULL,
	    &errmsg))
		errx(2, "search for initial revisions failed: %s", errmsg);
	if (sqlite3_exec(db, defaultbranch_statement, NULL, NULL, &errmsg))
		errx(2, "switch of .HEAD to default branch failed: %s", errmsg);
	if (sqlite3_exec(db, no_defaultbranch_statement, NULL, NULL, &errmsg))
		errx(2, "switch vendor branch files to .HEAD failed: %s", errmsg);

	if (sqlite3_exec(db, brachpoints_statement, NULL, NULL, &errmsg))
		errx(2, "updating parent branches failed: %s", errmsg);

	if (sqlite3_exec(db, parentless_branches, NULL, NULL, &errmsg))
		errx(2, "find branches without parent failed: %s", errmsg);

	if (sqlite3_prepare(db, next_unchecked_branches, -1, &next_stmt, NULL))
		errx(2, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, process_unchecked_branches, -1, &process_stmt,
	    NULL))
		errx(2, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, process_unchecked_branches2, -1,
	    &process_stmt2, NULL))
		errx(2, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));


	for (;;) {
		rv = sqlite3_step(next_stmt);
		if (rv == SQLITE_DONE)
			break;
		if (rv != SQLITE_ROW)
			errx(2, "sqlite3_step failed: %s", sqlite3_errmsg(db));
		id = sqlite3_column_int64(next_stmt, 0);
		if (sqlite3_reset(next_stmt))
			errx(2, "sqlite3_reset failed: %s", sqlite3_errmsg(db));
		if (sqlite3_bind_int64(process_stmt, 1, id))
			errx(2, "sqlite3_bind_int64 failed: %s",
			    sqlite3_errmsg(db));
		if (sqlite3_step(process_stmt) != SQLITE_DONE)
			errx(2, "sqlite3_step failed: %s", sqlite3_errmsg(db));
		if (sqlite3_reset(process_stmt))
			errx(2, "sqlite3_reset failed: %s", sqlite3_errmsg(db));

		if (sqlite3_bind_int64(process_stmt2, 1, id))
			errx(2, "sqlite3_bind_int64 failed: %s",
			    sqlite3_errmsg(db));
		if (sqlite3_bind_int64(process_stmt2, 2, id))
			errx(2, "sqlite3_bind_int64 failed: %s",
			    sqlite3_errmsg(db));
		if (sqlite3_step(process_stmt2) != SQLITE_DONE)
			errx(2, "sqlite3_step failed: %s", sqlite3_errmsg(db));
		if (sqlite3_reset(process_stmt2))
			errx(2, "sqlite3_reset failed: %s", sqlite3_errmsg(db));
	}
	sqlite3_finalize(next_stmt);
	sqlite3_finalize(process_stmt);
	sqlite3_finalize(process_stmt2);

	if (sqlite3_exec(db, branchpoints_warning_statement, branch_warning,
	    NULL, &errmsg))
		errx(2, "checking parent branches failed: %s", errmsg);

	if (sqlite3_exec(db, "COMMIT", NULL, 0, &errmsg))
		errx(1, "commit failed: %s", errmsg);

	if (sqlite3_close(db))
		errx(1, "db close failed");
	return 0;
}
