#include <sys/queue.h>
#include <assert.h>
#include <err.h>
#include <inttypes.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sqlite3.h>
#include <time.h>
#include <zlib.h>

#include <openssl/md5.h>
#include <openssl/sha.h>

#include "common.h"
#include "delta.h"

static int terminate;

/*
 * manifests size is determined by:
 * B card: 3 + 40 [may be skipped]
 * C card: 3 + quoted message
 * D card: 3 + 19 (ISO timestamp)
 * F cards: (4 + 40) * files + sum of path names
 * P card: 3 + 40
 * R card omitted, too expensive
 * T cards: "T *branch * name" "T *sym-name *" "T -sym-parent *" => 3 * symbol name + 33
 * U card: 3 + author name
 * Z card: 3 + 32
 *
 * Overall sum:
 * 106 + quoted message + author name + 44 * files + sum of path names + T cards
 */

static int *manifest_file_map;
static char **manifest_path;
static char *manifest_digest;
static char *manifest_mode;
static int32_t *cur_last_seen, cur_commit, cur_baseline;

static char *last_author;
static char date_limit[20];
static uint32_t files, message_limit, author_limit;
static uint32_t last_message_hash, changes_since_baseline, last_baseline_len;
static char last_manifest_id[41], base_manifest_id[41];
static char last_baseline[41];

static char branch_date[20];
static char *branch_name, *branch_parent;

static sqlite3 *db;
	sqlite3 *outdb;
static sqlite3_stmt *stmt, *hash_stmt, *insert_stmt, *delta_insert_stmt_main,
    *update_stmt_main, *branch_stmt, *branch_u_stmt, *check_uuid_stmt,
    *find_delta_stmt;
static int revisions_imported, revisions_total, permil;

#define DIGEST(i)	(manifest_digest + (i) * 41)

struct branch_todo {
	struct branch_todo *next;
	sqlite_int64 branch_id;
} *branch_todo;

static void
next_branch(void)
{
	if (sqlite3_step(branch_stmt) == SQLITE_ROW) {
		if (sqlite3_column_bytes(branch_stmt, 1) != 19)
			errx(1, "Inconsistent branch time");
		strcpy(branch_date,
		    (const char *)sqlite3_column_text(branch_stmt, 1));
	} else {
		strcpy(branch_date, "9999-99-99 99:99:99");
	}
}

struct buf {
	char *data;
	char *ptr;
	size_t free;
	size_t len;
	size_t pre_free;
	u_int refcount;
};

static struct buf *
buffer_new(size_t initial, size_t prefix)
{
	struct buf *bp;

	if (initial < prefix)
		initial = prefix;

	bp = malloc(sizeof(*bp));
	bp->data = malloc(initial);
	bp->ptr = bp->data + prefix;
	bp->free = initial - prefix;
	bp->len = initial;
	bp->pre_free = prefix;
	bp->refcount = 1;

	return bp;
}

static struct buf *
buffer_from_data(void *data, size_t len)
{
	struct buf *bp;

	bp = malloc(sizeof(*bp));
	bp->data = data;
	bp->len = len;
	bp->free = 0;
	bp->pre_free = 0;
	bp->ptr = bp->data + bp->len;
	bp->refcount = 1;

	return bp;
}

static void
buffer_reference(struct buf *bp)
{
	__sync_add_and_fetch(&bp->refcount, 1);
}

static void
buffer_free(struct buf *bp)
{
	if (__sync_sub_and_fetch(&bp->refcount, 1) == 0) {
		free(bp->data);
		free(bp);
	}
}

static void
buffer_resize(struct buf *bp, size_t increment)
{
	size_t idx;

	if (increment < 16384)
		increment = 16384;

	bp->free += increment;
	bp->len += increment;
	idx = bp->ptr - bp->data;
	bp->data = realloc(bp->data, bp->len);
	bp->ptr = bp->data + idx;
}

static void
buffer_put(struct buf *bp, char c)
{
	if (!bp->free)
		buffer_resize(bp, 1);
	*bp->ptr++ = c;
	bp->free--;
}

static void
buffer_prepend(struct buf *bp, const void *data, size_t len)
{
	assert(bp->pre_free >= len); /* XXX lazy... */
	memcpy(bp->data + bp->pre_free - len, data, len);
	bp->pre_free -= len;
}

static void
buffer_append(struct buf *bp, const void *data, size_t len)
{
	if (bp->free < len)
		buffer_resize(bp, len);
	memcpy(bp->ptr, data, len);
	bp->ptr += len;
	bp->free -= len;
}

static void
buffer_cat(struct buf *bp, const char *str)
{
	buffer_append(bp, str, strlen(str));
}

static size_t
buffer_len(struct buf *bp)
{
	return bp->ptr - bp->data - bp->pre_free;
}

static const void *
buffer_data(struct buf *bp)
{
	return bp->data + bp->pre_free;
}

static void
digest2hex(char *text, const unsigned char *digest, size_t len)
{
	static const char hex[16] = "0123456789abcdef";

	while (len--) {
		*text++ = hex[*digest / 16];
		*text++ = hex[*digest++ % 16];
	}
}

LIST_HEAD(, compress_queue) compress_queue =
    LIST_HEAD_INITIALIZER(&compress_queue);
int compress_queue_len;
int compress_queue_limit = -1;
pthread_mutex_t compress_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t compress_queue_busy =  PTHREAD_COND_INITIALIZER;
pthread_cond_t compress_queue_limited =  PTHREAD_COND_INITIALIZER;

struct compress_queue {
	LIST_ENTRY(compress_queue) link;
	struct buf *bp;
	char uuid[40];
};

static void
add_compress_queue(struct buf *bp, const char uuid[40])
{
	struct compress_queue *q;

	q = malloc(sizeof(*q));
	buffer_reference(bp);
	q->bp = bp;
	memcpy(q->uuid, uuid, 40);
	pthread_mutex_lock(&compress_queue_mutex);
	LIST_INSERT_HEAD(&compress_queue, q, link);
	while (compress_queue_limit != -1 &&
	    compress_queue_len >= compress_queue_limit)
		pthread_cond_wait(&compress_queue_limited,
		    &compress_queue_mutex);
	if (compress_queue_len == 0)
		pthread_cond_signal(&compress_queue_busy);
	++compress_queue_len;
	pthread_mutex_unlock(&compress_queue_mutex);
}

static void *
process_compress_queue(void *arg)
{
	sqlite3_stmt *update_stmt = arg;
	struct compress_queue *q;
	uLongf olen;
	unsigned char *cb;
	size_t len;

	for (;;) {
		pthread_mutex_lock(&compress_queue_mutex);
		while ((q = LIST_FIRST(&compress_queue)) == NULL) {
			if (terminate < 2) {
				pthread_cond_wait(&compress_queue_busy, &compress_queue_mutex);
				continue;
			}
			pthread_mutex_unlock(&compress_queue_mutex);
			return NULL;
		}
		LIST_REMOVE(q, link);
		--compress_queue_len;
		if (compress_queue_len <= compress_queue_limit / 2)
			pthread_cond_signal(&compress_queue_limited);
		pthread_mutex_unlock(&compress_queue_mutex);

		len = buffer_len(q->bp);
		olen = len + len / 100 + 12;
		cb = malloc(olen + 4);
		if (compress2(cb + 4, &olen, buffer_data(q->bp), len, 9) != Z_OK)
			errx(1, "compress failed");
		olen += 4;
		cb[0] = (len >> 24) & 0xff;
		cb[1] = (len >> 16) & 0xff;
		cb[2] = (len >> 8) & 0xff;
		cb[3] = len & 0xff;

		if (sqlite3_bind_blob(update_stmt, 1, cb, olen, free))
			errx(1, "sqlite3_bind_blob failed");
		if (sqlite3_bind_text(update_stmt, 2, q->uuid, 40,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text_failed");
		if (sqlite3_step(update_stmt) != SQLITE_DONE)
			errx(1, "sqlite3_step for delta update failed");
		sqlite3_reset(update_stmt);
		buffer_free(q->bp);
		free(q);
	}
}

LIST_HEAD(, delta_queue) delta_queue = LIST_HEAD_INITIALIZER(&zlib_queue);
int delta_queue_len;
int delta_queue_limit = -1;
pthread_mutex_t delta_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t delta_queue_busy =  PTHREAD_COND_INITIALIZER;
pthread_cond_t delta_queue_limited =  PTHREAD_COND_INITIALIZER;

struct delta_queue {
	LIST_ENTRY(delta_queue) link;
	struct buf *before;
	struct buf *after;
	char before_uuid[40];
	char after_uuid[40];
};

static void
add_delta_queue(struct buf *before, const char before_uuid[40],
    struct buf *after, const char after_uuid[40])
{
	struct delta_queue *q;

	q = malloc(sizeof(*q));
	buffer_reference(before);
	q->before = before;
	memcpy(q->before_uuid, before_uuid, 40);
	buffer_reference(after);
	q->after = after;
	memcpy(q->after_uuid, after_uuid, 40);
	pthread_mutex_lock(&delta_queue_mutex);
	LIST_INSERT_HEAD(&delta_queue, q, link);
	while (delta_queue_limit != -1 &&
	    delta_queue_len >= delta_queue_limit)
		pthread_cond_wait(&delta_queue_limited,
		    &delta_queue_mutex);
	if (delta_queue_len == 0)
		pthread_cond_signal(&delta_queue_busy);
	++delta_queue_len;
	pthread_mutex_unlock(&delta_queue_mutex);
}

static void *
process_delta_queue(void *arg)
{
	sqlite3_stmt *delta_insert_stmt = arg;
	struct delta_queue *q;
	int delta_len;
	char *delta;
	struct buf *bp;

	for (;;) {
		pthread_mutex_lock(&delta_queue_mutex);
		while ((q = LIST_FIRST(&delta_queue)) == NULL) {
			if (terminate < 1) {
				pthread_cond_wait(&delta_queue_busy, &delta_queue_mutex);
				continue;
			}
			pthread_mutex_unlock(&delta_queue_mutex);
			return NULL;
		}
		LIST_REMOVE(q, link);
		--delta_queue_len;
		if (delta_queue_len <= delta_queue_limit / 2)
			pthread_cond_signal(&delta_queue_limited);
		pthread_mutex_unlock(&delta_queue_mutex);

		delta = malloc(buffer_len(q->before) + 60);
		delta_len = delta_create(buffer_data(q->after),
		    buffer_len(q->after), buffer_data(q->before),
		    buffer_len(q->before), delta);

		if (buffer_len(q->before) * 3 < delta_len * 4) {
			free(delta);
			add_compress_queue(q->before, q->before_uuid);
			buffer_free(q->before);
			buffer_free(q->after);
			free(q);
			continue;
		}

		bp = buffer_from_data(delta, delta_len);
		add_compress_queue(bp, q->before_uuid);
		buffer_free(bp);

		if (sqlite3_bind_text(delta_insert_stmt, 1, q->before_uuid, 40,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text_failed");
		if (sqlite3_bind_text(delta_insert_stmt, 2, q->after_uuid, 40,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text_failed");
		if (sqlite3_step(delta_insert_stmt) != SQLITE_DONE)
			errx(1, "sqlite3_step for delta failed");
		sqlite3_reset(delta_insert_stmt);

		buffer_free(q->before);
		buffer_free(q->after);
		free(q);
	}
}

static struct buf *pending_buffer;
static char pending_manifest[40];
static int pending_delta;

static void
flush_pending_manifest(void)
{
	if (pending_buffer) {
		add_compress_queue(pending_buffer, pending_manifest);
		buffer_free(pending_buffer);
		pending_buffer = NULL;
	}
}

static void
write_manifest(struct buf *bp, int use_delta, const char *manifest_id,
    const char *old_manifest_id)
{

	if (sqlite3_bind_text(insert_stmt, 1, manifest_id, 40,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_bind_int64(insert_stmt, 2, buffer_len(bp)))
		errx(1, "sqlite3_bind_int64 failed");

	if (sqlite3_step(insert_stmt) != SQLITE_DONE)
		errx(1, "sqlite3_step for content failed");
	sqlite3_reset(insert_stmt);

	if (buffer_len(bp) < 50 || pending_buffer == NULL ||
	    memcmp(old_manifest_id, pending_manifest, 40) ||
	    (pending_delta && !use_delta)) {
		flush_pending_manifest();
	} else {
		add_delta_queue(pending_buffer, old_manifest_id, bp,
		    manifest_id);
		buffer_free(pending_buffer);
	}
	pending_buffer = bp;
	memcpy(pending_manifest, manifest_id, 40);
	pending_delta = use_delta;
}

static void
buffer_append_quote(struct buf *bp, const char *data)
{
	size_t skip;

	for (; *data; ++data) {
		skip = strcspn(data, "\\\r \t\n\f\v");
		if (skip)
			buffer_append(bp, data, skip);
		data += skip;
		if (*data == '\0')
			break;

		switch (*data) {
		case '\0':
			buffer_append(bp, "\\0", 2);
			break;
		case '\\':
			buffer_append(bp, "\\\\", 2);
			break;
		case '\r':
			buffer_append(bp, "\\r", 2);
			break;
		case ' ':
			buffer_append(bp, "\\s", 2);
			break;
		case '\t':
			buffer_append(bp, "\\t", 2);
			break;
		case '\n':
			buffer_append(bp, "\\n", 2);
			break;
		case '\f':
			buffer_append(bp, "\\f", 2);
			break;
		case '\v':
			buffer_append(bp, "\\v", 2);
			break;
		default:
			abort();
		}
	}
}

static void
finish_manifest(struct buf *bp, const char *date, int flush)
{	MD5_CTX ctx;
	SHA_CTX ctx2;
	unsigned char md5_digest[MD5_DIGEST_LENGTH];
	char md5_digest_hex[2 * MD5_DIGEST_LENGTH];
	unsigned char sha1_digest[SHA_DIGEST_LENGTH];
	char sha1_digest_hex[2 * SHA_DIGEST_LENGTH];
	uint32_t changed_files;
	unsigned int i;
	int use_delta;

	if (cur_baseline == 0 || last_baseline_len < changes_since_baseline * 125)
		use_delta = 0;
	else
		use_delta = 1;

	if (use_delta) {
		buffer_prepend(bp, "\n", 1);
		buffer_prepend(bp, last_baseline, 40);
		buffer_prepend(bp, "B ", 2);
	} else {
		cur_baseline = cur_commit;
	}

	for (changed_files = 0, i = 0; i < files; ++i) {
		if (!use_delta && DIGEST(i)[0] == '\0')
			continue;
		if (use_delta && cur_last_seen[i] <= cur_baseline)
			continue;
		++changed_files;
		buffer_append(bp, "F ", 2);
		buffer_append_quote(bp, manifest_path[i]);
		if (DIGEST(i)[0]) {
			buffer_put(bp, ' ');
			buffer_append(bp, DIGEST(i), 40);
			if (manifest_mode[i])
				buffer_append(bp, " x", 2);
		}
		buffer_put(bp, '\n');
	}
	if (*last_manifest_id) {
		buffer_append(bp, "P ", 2);
		buffer_append(bp, last_manifest_id, 40);
		buffer_put(bp, '\n');
	}
	/* R card is not computed */
	/* Add T record for branch commit */
	if (branch_name) {
		buffer_cat(bp, "T *branch * ");
		buffer_cat(bp, branch_name);
		buffer_put(bp, '\n');
		buffer_cat(bp, "T *sym-");
		buffer_cat(bp, branch_name);
		buffer_append(bp, " *\n", 3);
		free(branch_name);
		branch_name = NULL;
	}
	if (branch_parent) {
		buffer_cat(bp, "T -sym-");
		buffer_cat(bp, branch_parent);
		buffer_append(bp, " *\n", 3);
		free(branch_parent);
		branch_parent = NULL;
	}
	buffer_append(bp, "U ", 2);
	buffer_cat(bp, last_author);
	buffer_put(bp, '\n');

	MD5_Init(&ctx);
	MD5_Update(&ctx, buffer_data(bp), buffer_len(bp));
	MD5_Final(md5_digest, &ctx);
	buffer_append(bp, "Z ", 2);
	digest2hex(md5_digest_hex, md5_digest, MD5_DIGEST_LENGTH);
	buffer_append(bp, md5_digest_hex, 2 * MD5_DIGEST_LENGTH);
	buffer_put(bp, '\n');

	SHA1_Init(&ctx2);
	SHA1_Update(&ctx2, buffer_data(bp), buffer_len(bp));
	SHA1_Final(sha1_digest, &ctx2);
	digest2hex(sha1_digest_hex, sha1_digest, SHA_DIGEST_LENGTH);

	write_manifest(bp, use_delta, sha1_digest_hex, last_manifest_id);

	memcpy(last_manifest_id, sha1_digest_hex, 2 * SHA_DIGEST_LENGTH);
	last_manifest_id[2 * SHA_DIGEST_LENGTH] = '\0';

	if (!use_delta) {
		strcpy(last_baseline, last_manifest_id);
		changes_since_baseline = 0;
		last_baseline_len = changed_files;
	}

	while (flush || strcmp(date, branch_date) >  0) {
		struct branch_todo *todo;

		if (branch_date[0] == '9')
			break;
		if (sqlite3_bind_text(branch_u_stmt, 1, last_manifest_id, -1,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text failed");
		if (sqlite3_bind_int64(branch_u_stmt, 2,
		    sqlite3_column_int64(branch_stmt, 0)))
			errx(1, "sqlite3_bind_int64 failed");
		if (sqlite3_step(branch_u_stmt) != SQLITE_DONE)
			errx(1, "sqlite3_step1 failed: %s", sqlite3_errmsg(db));
		sqlite3_reset(branch_u_stmt);

		todo = malloc(sizeof(*todo));
		todo->next = branch_todo;
		todo->branch_id = sqlite3_column_int64(branch_stmt, 0);
		branch_todo = todo;

		next_branch();
	}
}

static struct buf *
prepare_manifest(const char *commit_msg, const char *date, const char *author)
{
	struct buf *bp;

	bp = buffer_new(1024, 43);

	buffer_append(bp, "C ", 2);
	buffer_append_quote(bp, commit_msg);
	buffer_append(bp, "\nD ", 3);
	buffer_append(bp, date, 10);
	buffer_put(bp, 'T');
	buffer_append(bp, date + 11, 8);
	buffer_append(bp, ".000\n", 5);

	strcpy(last_author, author);

	return bp;
}

static void
add_time(const char *input, char *output, size_t output_len, int secs)
{
	struct tm tm;

	if (secs >= 3600)
		errx(1, "programming error");

	if (strptime(input, "%Y-%m-%d %H:%M:%S", &tm) != input + 19)
		errx(1, "Parsing date `%s' failed", input);

	tm.tm_sec += secs;
	tm.tm_min += tm.tm_sec / 60;
	tm.tm_sec %= 60;
	if (tm.tm_min >= 60) {
		if (++tm.tm_hour >= 24) {
			tm.tm_min -= 24;
			++tm.tm_mday;
			switch (tm.tm_mday) {
			case 32:
				/* Assume at most one day passed */
				tm.tm_mday = 1;
				++tm.tm_mon;
				break;
			case 31:
				switch (tm.tm_mon) {
				case 4:
				case 6:
				case 9:
				case 11:
					tm.tm_mday = 1;
					++tm.tm_mon;
					break;
				}
				break;
			case 30:
				if (tm.tm_year % 4 == 0) {
					tm.tm_mday = 1;
					++tm.tm_mon;
				}
				break;
			case 29:
				if (tm.tm_year % 4 != 0) {
					tm.tm_mday = 1;
					++tm.tm_mon;
				}
				break;
			}
			if (tm.tm_mon == 13) {
				tm.tm_mon = 1;
				++tm.tm_year;
			}
		}
	}
	strftime(output, output_len, "%Y-%m-%d %H:%M:%S", &tm);
}

static struct buf *
flush_manifest(struct buf *bp)
{
	const char *date;

	date = (const char *)sqlite3_column_text(stmt, 2);

	if (cur_commit > 2)
		finish_manifest(bp, date, 0);

	if (sqlite3_column_bytes(stmt, 2) != 19)
		errx(1, "inconsistent database2");

	bp = prepare_manifest((const char *)sqlite3_column_text(stmt, 3),
	    date, (const char *)sqlite3_column_text(stmt, 1));

	add_time(date, date_limit, sizeof(date_limit), 600);

	++cur_commit;
	return bp;
}

static void
mark_file(sqlite_int64 file, sqlite_int64 content)
{
	char old_hash[41];

	if (cur_last_seen[file] <= cur_baseline)
		++changes_since_baseline;
	cur_last_seen[file] = cur_commit;

	if (content == 0) {
		DIGEST(file)[0] = '\0';
		return;
	}

	if (sqlite3_bind_int64(hash_stmt, 1, content))
		errx(1, "sqlite3_bind_int64 failed");
	switch (sqlite3_step(hash_stmt)) {
	case SQLITE_ROW:
		break;
	case SQLITE_DONE:
		errx(1, "Hash not found for %lld", (long long)content);
	default:
		errx(1, "sqlite3_step failed: %s", sqlite3_errmsg(db));
	}

	if (sqlite3_column_bytes(hash_stmt, 0) != 40)
		errx(1, "Incorrect size of hash");
	strcpy(old_hash, DIGEST(file));
	strcpy(DIGEST(file), (const char *)sqlite3_column_text(hash_stmt, 0));

	if (sqlite3_bind_text(check_uuid_stmt, 1, DIGEST(file), -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_step(check_uuid_stmt) == SQLITE_ROW) {
		sqlite3_reset(check_uuid_stmt);
		sqlite3_reset(hash_stmt);
		return;
	}
	sqlite3_reset(check_uuid_stmt);

	if (sqlite3_bind_text(insert_stmt, 1, DIGEST(file), -1, SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_bind_int64(insert_stmt, 2,
	    sqlite3_column_int64(hash_stmt, 1)))
		errx(1, "sqlite3_bind_int64 failed");

	if (sqlite3_step(insert_stmt) != SQLITE_DONE)
		errx(1, "sqlite3_step for content failed");

	sqlite3_reset(insert_stmt);
	sqlite3_reset(hash_stmt);

	if (sqlite3_bind_text(find_delta_stmt, 1, DIGEST(file), -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_bind_text(find_delta_stmt, 2, old_hash, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_step(find_delta_stmt) != SQLITE_ROW) {
		sqlite3_reset(find_delta_stmt);
		return;
	}

	if (sqlite3_bind_text(delta_insert_stmt_main, 1, old_hash, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text_failed");
	if (sqlite3_bind_text(delta_insert_stmt_main, 2, DIGEST(file), -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text_failed");
	if (sqlite3_step(delta_insert_stmt_main) != SQLITE_DONE)
		errx(1, "sqlite3_step for delta file failed");
	sqlite3_reset(delta_insert_stmt_main);

	if (sqlite3_bind_blob(update_stmt_main, 1,
	    sqlite3_column_blob(find_delta_stmt, 0),
	    sqlite3_column_bytes(find_delta_stmt, 0), SQLITE_STATIC))
		errx(1, "sqlite3_bind_blob failed");
	if (sqlite3_bind_text(update_stmt_main, 2, old_hash, -1,
	    SQLITE_STATIC))
		errx(1, "sqlite3_bind_text_failed: %s", sqlite3_errmsg(db));
	if (sqlite3_step(update_stmt_main) != SQLITE_DONE)
		errx(1, "sqlite3_step for delta update failed");
	sqlite3_reset(update_stmt_main);
	sqlite3_reset(find_delta_stmt);
}

static struct buf *
process_row(struct buf *bp)
{
	/* file,author,date,message,content */
	sqlite_int64 file = manifest_file_map[sqlite3_column_int64(stmt, 0)];
	sqlite_int64 content = sqlite3_column_int64(stmt, 4);

	uint32_t message_hash = crc32(0, sqlite3_column_text(stmt, 3),
	    sqlite3_column_bytes(stmt, 3));
	if (message_hash != last_message_hash ||
	    cur_last_seen[file] == cur_commit ||
	    strcmp(last_author, (const char *)sqlite3_column_text(stmt, 1)) ||
	    strcmp(date_limit, (const char *)sqlite3_column_text(stmt, 2)) < 0)
		bp = flush_manifest(bp);

	last_message_hash = message_hash;

	mark_file(file, content);

	return bp;
}

static void
process_branch(sqlite_int64 branch_id, int vendor)
{
	struct buf *bp;
	int cur_permil;
	sqlite3_int64 parent;
	char *msg;
	const char *tmp_branch_name, *revisions;
	sqlite3_stmt *name_stmt, *branchtime_stmt, *parent_stmt, *manifest_stmt;

	if (sqlite3_prepare(db, "SELECT symbol FROM branches WHERE id=?", -1,
	    &name_stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_bind_int64(name_stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_prepare(db, "SELECT parent FROM branchpoints2 "
	    "WHERE branch=?", -1, &parent_stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_bind_int64(parent_stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_prepare(db, "SELECT date FROM branchtime WHERE branch=?",
	    -1, &branchtime_stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_bind_int64(branchtime_stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_prepare(db, "SELECT hash FROM branchmanifest WHERE branch=?",
	    -1, &manifest_stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_bind_int64(manifest_stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");


	if (sqlite3_step(manifest_stmt) == SQLITE_ROW) {
		if (sqlite3_column_bytes(manifest_stmt, 0) != 40)
			errx(1, "manifest hash has wrong length");
		strcpy(last_manifest_id,
		    (const char *)sqlite3_column_text(manifest_stmt, 0));
	} else {
		strcpy(last_manifest_id, base_manifest_id);
	}

	if (sqlite3_step(name_stmt) != SQLITE_ROW)
		errx(1, "Branch name not found");

	tmp_branch_name = (const char *)sqlite3_column_text(name_stmt, 0);
	if (strcmp(tmp_branch_name, ".HEAD"))
		branch_name = strdup(tmp_branch_name);
	else
		branch_name = strdup("trunk");

	switch (sqlite3_step(parent_stmt)) {
	case SQLITE_ROW:
		parent = sqlite3_column_int64(parent_stmt, 0);
		break;
	case SQLITE_DONE:
		parent = 0;
		break;
	default:
		errx(1, "Obtaining parent branch failed: %s",
		    sqlite3_errmsg(db));
	}

	switch (sqlite3_step(branchtime_stmt)) {
	case SQLITE_ROW:
		strcpy(date_limit,
		    (const char *)sqlite3_column_text(branchtime_stmt, 0));
		break;
	case SQLITE_DONE:
		*date_limit = '\0';
		break;
	default:
		errx(1, "Obtaining branch time failed: %s",
		    sqlite3_errmsg(db));
	}

	sqlite3_finalize(manifest_stmt);
	sqlite3_finalize(branchtime_stmt);
	sqlite3_finalize(parent_stmt);
	sqlite3_finalize(name_stmt);

	if (parent) {
		const char *parent_name;
	
		if (sqlite3_prepare(db, "SELECT symbol FROM branches WHERE id=?",
		    -1, &stmt, NULL))
			errx(1, "sqlite3_prepare9 failed");
		if (sqlite3_bind_int64(stmt, 1, parent))
			errx(1, "sqlite3_bind_text failed");

		if (sqlite3_step(stmt) != SQLITE_ROW)
			errx(1, "inconsistent database");
		if (strcmp((const char *)sqlite3_column_text(stmt, 0), ".HEAD"))
			parent_name = (const char *)sqlite3_column_text(stmt,
			    0);
		else
			parent_name = "trunk";
		branch_parent = strdup(parent_name);

		sqlite3_finalize(stmt);
	}

	if (sqlite3_prepare(db, "SELECT hash,size,content FROM content "
	    "WHERE id=?", -1, &hash_stmt, NULL))
		errx(1, "sqlite3_prepare5 failed");

	if (sqlite3_prepare(db, "SELECT delta FROM delta "
	    "WHERE hash=? AND base=?", -1, &find_delta_stmt, NULL))
		errx(1, "sqlite3_prepare5 failed");

	if (sqlite3_prepare(db, "SELECT DISTINCT b.id, bt.date "
	    "FROM branches b, branchtime bt, branchpoints2 bp "
	    "WHERE b.id=bt.branch AND bp.branch=b.id AND bp.parent=? "
	    "AND NOT b.vendorbranch ORDER BY bt.date, b.symbol",
	    -1, &branch_stmt, NULL))
		errx(1, "sqlite3_prepare7 failed");
	if (sqlite3_bind_int64(branch_stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");

	if (sqlite3_prepare(db, "INSERT INTO branchmanifest VALUES (?,?)", -1,
	    &branch_u_stmt, NULL))
		errx(1, "sqlite3_prepare8 failed");

	cur_baseline = 0;
	memset(cur_last_seen, 0, 4 * files);
	memset(manifest_digest, 0, 41 * files);
	*last_author = 0;

	if (sqlite3_prepare(db, "SELECT r.file, r.content FROM revision r, "
	    "branchpoints2 bp WHERE bp.branch=? AND r.id=bp.revision",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_bind_int64(stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");

	if (!vendor && branch_name && sqlite3_step(stmt) == SQLITE_ROW) {
		cur_commit = 1;

		do {
			sqlite_int64 file =
			    manifest_file_map[sqlite3_column_int64(stmt, 0)];
			sqlite_int64 content = sqlite3_column_int64(stmt, 1);
			mark_file(file, content);
		} while (sqlite3_step(stmt) == SQLITE_ROW);
		++cur_commit;
	}
	sqlite3_finalize(stmt);

	cur_commit = 2;

	next_branch();

	if (vendor) {
		revisions = "SELECT DISTINCT r.file,r.author,r.date,r.message, "
		    "r.content FROM revision r, revbranches4 rb, file f "
		    "WHERE r.id=rb.revision AND f.id=rb.file AND rb.branch=? "
		    "ORDER BY rb.date,f.path";
	} else {
		revisions = "SELECT r.file,r.author,r.date,r.message, "
		    "r.content FROM revision r, revbranches3 rb, file f "
		    "WHERE r.id=rb.revision AND f.id=rb.file AND rb.branch=? "
		    "ORDER BY rb.date,f.path";
	}
	if (sqlite3_prepare(db, revisions, -1, &stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_bind_int64(stmt, 1, branch_id))
		errx(1, "sqlite3_bind_text failed");

	bp = NULL;

	while (sqlite3_step(stmt) == SQLITE_ROW) {
		bp = process_row(bp);
		cur_permil = revisions_imported++ * 1000 / revisions_total;
		if (cur_permil > 1000)
			cur_permil = 1000;

		if (cur_permil != permil) {
			permil = cur_permil;
			printf("\r%d.%d%%", cur_permil / 10,
			    cur_permil % 10);
			fflush(stdout);
		}
	}
	sqlite3_finalize(stmt);

	if (cur_commit == 2) {
		asprintf(&msg, "Creating branch %s", branch_name);
		add_time(date_limit, date_limit, sizeof(date_limit), 1);
		bp = prepare_manifest(msg, date_limit, "cvs");
		free(msg);

	}
	finish_manifest(bp, NULL, 1);

	sqlite3_finalize(branch_u_stmt);
	sqlite3_finalize(branch_stmt);
	sqlite3_finalize(hash_stmt);
	sqlite3_finalize(find_delta_stmt);
}

#ifdef SQL_DEBUG
static void
tracef(void *unused, const char *sql)
{
	fprintf(stderr, "%s;\n", sql);
}
#endif

static void
usage(void)
{
	fprintf(stderr, "Usage: %s [-b 40-byte-hash] database repository\n",
	    getprogname());
	exit(1);
}

static const char init_statement[] =
    "DROP TABLE IF EXISTS revbranches3;\n"
    "DROP TABLE IF EXISTS revbranches4;\n"
    "CREATE TABLE revbranches3 "
    "(revision integer NOT NULL REFERENCES revision(id), "
    "branch integer NOT NULL REFERENCES branches(id),"
    "file integer NOT NULL REFERENCES file(id),"
    "date datetime NOT NULL);\n"
    "CREATE TABLE revbranches4 "
    "(revision integer NOT NULL REFERENCES revision(id), "
    "branch integer NOT NULL REFERENCES branches(id),"
    "file integer NOT NULL REFERENCES file(id),"
    "date datetime NOT NULL,"
    "UNIQUE(revision, branch));\n"
    "CREATE INDEX revbranches3_bd ON revbranches3(branch,date);\n"
    "CREATE INDEX revbranches4_fd ON revbranches4(branch, date);\n"
    "INSERT INTO revbranches3 SELECT r.id, rb.branch, r.file, r.date "
    "FROM revision r, revbranches2 rb WHERE r.id=rb.revision AND "
    "NOT EXISTS (SELECT * FROM skiprev2 WHERE revision=r.id);\n"
    "INSERT INTO revbranches4 SELECT r.id, rb.branch, r.file, r.date "
    "FROM revision r, revbranches rb, branches b WHERE r.id=rb.revision AND "
    "rb.branch=b.id AND b.vendorbranch;\n"
    "INSERT INTO revbranches4 SELECT rb.revision, b.id, rb.file, rb.date "
    "FROM revbranches4 rb, branches b, symbol s WHERE rb.file=s.file AND "
    "s.revision=\"1.1.1\" AND b.symbol=s.symbol AND b.id != rb.branch";

int
main(int argc, char **argv)
{
	struct branch_todo *todo;
	char *errmsg;
	sqlite3_stmt *vendor_stmt, **update_stmt_thread, **delta_stmt_thread;
	sqlite3_stmt *stmt, *stmt2;
	int ch, i, j;
	long ncpu;
	pthread_t *threads;

	ncpu = sysconf(_SC_NPROCESSORS_ONLN);
	update_stmt_thread = calloc(sizeof(*update_stmt_thread), ncpu);
	delta_stmt_thread = calloc(sizeof(*delta_stmt_thread), ncpu);
	threads = calloc(sizeof(*threads), 2 * ncpu);

	while ((ch = getopt(argc, argv, "b:l:")) != -1) {
		switch (ch) {
		case 'b':
			if (strlen(optarg) != 40)
				usage();
			strcpy(base_manifest_id, optarg);
			break;
		case 'l':
			compress_queue_limit = atoi(optarg);
			delta_queue_limit = atoi(optarg);
			break;
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 2)
		usage();

	if (sqlite3_open_v2(argv[0], &db, SQLITE_OPEN_READWRITE, NULL))
		errx(1, "database open failed: %s", sqlite3_errmsg(db));

#ifdef SQL_DEBUG
	sqlite3_trace(db, tracef, NULL);
#endif

	if (sqlite3_open_v2(argv[1], &outdb, SQLITE_OPEN_READWRITE, NULL))
		errx(1, "database open failed: %s", sqlite3_errmsg(outdb));

	sqlite3_exec(db, "BEGIN EXCLUSIVE", NULL, 0, NULL);
	sqlite3_exec(db, "PRAGMA cache_size=16384", NULL, 0, NULL);
	sqlite3_exec(db, "PRAGMA synchronous = OFF", NULL, 0, NULL);

	sqlite3_exec(outdb, "PRAGMA cache_size = 32768", NULL, 0, NULL);
	sqlite3_exec(outdb, "PRAGMA synchronous = OFF", NULL, 0, NULL);
	sqlite3_exec(outdb, "PRAGMA journal_mode = OFF", NULL, 0, NULL);
	sqlite3_exec(outdb, "BEGIN EXCLUSIVE", NULL, 0, NULL);

	if (sqlite3_exec(db, "DROP TABLE IF EXISTS branchmanifest;"
	    "CREATE TABLE branchmanifest (hash varchar, "
	    "branch integer REFERENCES branches(id) UNIQUE)", NULL, 0, &errmsg))
		errx(1, "schema update failed: %s", errmsg);
	if (sqlite3_exec(db, init_statement, NULL, 0, &errmsg))
		errx(1, "schema update failed: %s", errmsg);

	if (sqlite3_prepare(outdb, "SELECT uuid FROM blob WHERE uuid=?",
	    -1, &check_uuid_stmt, NULL))
		errx(1, "sqlite3_prepare6 failed");

	if (sqlite3_prepare(outdb, "INSERT OR REPLACE INTO blob "
	    "(rcvid,uuid,size,content) VALUES (0,?,?,NULL)", -1,
	    &insert_stmt, NULL))
		errx(1, "sqlite3_prepare6 failed");

	if (sqlite3_prepare(outdb, "INSERT OR REPLACE INTO delta (rid,srcid) "
	    "SELECT b.rid, b2.rid FROM blob b, blob b2 WHERE b.uuid=? AND b2.uuid=?", -1,
	    &delta_insert_stmt_main, NULL))
		errx(1, "sqlite3_prepare6 failed");

	for (i = 0; i < ncpu; ++i) {
		if (sqlite3_prepare(outdb, "INSERT OR REPLACE INTO delta (rid,srcid) "
		    "SELECT b.rid, b2.rid FROM blob b, blob b2 WHERE b.uuid=? AND b2.uuid=?", -1,
		    &delta_stmt_thread[i], NULL))
			errx(1, "sqlite3_prepare6 failed");
	}

	if (sqlite3_prepare(outdb, "UPDATE blob SET content=? WHERE uuid=?",
	    -1, &update_stmt_main, NULL))
		errx(1, "sqlite3_prepare6 failed");

	for (i = 0; i < ncpu; ++i) {
		if (sqlite3_prepare(outdb, "UPDATE blob SET content=? WHERE uuid=?",
		    -1, &update_stmt_thread[i], NULL))
			errx(1, "sqlite3_prepare6 failed");
	}

	if (sqlite3_prepare(db, "SELECT MAX(id) FROM file", -1, &stmt, NULL))
		errx(1, "sqlite3_prepare1 failed");
	if (sqlite3_step(stmt) != SQLITE_ROW)
		errx(1, "sqlite3_step failed3: %s", sqlite3_errmsg(db));
	files = sqlite3_column_int(stmt, 0) + 1;
	sqlite3_finalize(stmt);

	if (sqlite3_prepare(db, "SELECT MAX(LENGTH(author)), "
	    "MAX(LENGTH(message)) FROM revision", -1, &stmt, NULL))
		errx(1, "sqlite3_prepare2 failed");
	if (sqlite3_step(stmt) != SQLITE_ROW)
		errx(1, "sqlite3_step4 failed: %s", sqlite3_errmsg(db));
	author_limit = sqlite3_column_int(stmt, 0);
	last_author = malloc(author_limit + 1);
	message_limit = sqlite3_column_int(stmt, 1) * 2;
	sqlite3_finalize(stmt);

	if (sqlite3_prepare(db, "SELECT MAX(LENGTH(symbol)) FROM branches",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare10 failed");
	if (sqlite3_step(stmt) != SQLITE_ROW)
		errx(1, "sqlite3_step5 failed: %s", sqlite3_errmsg(db));
	message_limit += sqlite3_column_int(stmt, 0) * 3 + 33;
	sqlite3_finalize(stmt);

	if (sqlite3_prepare(db, "SELECT COUNT(*) FROM revision",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare10 failed");
	if (sqlite3_step(stmt) != SQLITE_ROW)
		errx(1, "sqlite3_step5 failed: %s", sqlite3_errmsg(db));
	revisions_total = sqlite3_column_int(stmt, 0);
	sqlite3_finalize(stmt);

	if (sqlite3_prepare(db, "SELECT COUNT(*) FROM revbranches4",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare10 failed");
	if (sqlite3_step(stmt) != SQLITE_ROW)
		errx(1, "sqlite3_step5 failed: %s", sqlite3_errmsg(db));
	revisions_total += sqlite3_column_int(stmt, 0);
	sqlite3_finalize(stmt);

	if (sqlite3_prepare(db, "SELECT COUNT(*) FROM skiprev2",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare10 failed");
	if (sqlite3_step(stmt) != SQLITE_ROW)
		errx(1, "sqlite3_step5 failed: %s", sqlite3_errmsg(db));
	revisions_total -= sqlite3_column_int(stmt, 0);
	sqlite3_finalize(stmt);

	if (revisions_total < 1)
		revisions_total = 1;

	manifest_digest = calloc(files, 41);
	manifest_file_map = calloc(files, 4);
	manifest_path = calloc(files, sizeof(char *));
	manifest_mode = calloc(files, 1);
	cur_last_seen = calloc(files, 4);

	if (sqlite3_prepare(db, "SELECT id,path,executable FROM file ORDER BY path",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare3 failed");
	j = 0;
	while (sqlite3_step(stmt) == SQLITE_ROW) {
		i = sqlite3_column_int(stmt, 0);
		if (i <= 0 || (unsigned)i >= files)
			errx(1, "consistency failed");
		manifest_file_map[i] = j;
		manifest_path[j] =
		    strdup((const char *)sqlite3_column_text(stmt, 1));
		manifest_mode[j] = sqlite3_column_int(stmt, 2);
		++j;
	}
	sqlite3_finalize(stmt);

	if (sqlite3_prepare(db, "SELECT b.id FROM branches b WHERE "
	    "NOT b.vendorbranch AND NOT EXISTS(SELECT * FROM branchpoints2 "
	    "WHERE branch=b.id)", -1, &stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));

	while (sqlite3_step(stmt) == SQLITE_ROW) {
		todo = malloc(sizeof(*todo));
		todo->next = branch_todo;
		todo->branch_id = sqlite3_column_int64(stmt, 0);
		branch_todo = todo;
	}
	sqlite3_finalize(stmt);

	for (i = 0; i < ncpu; ++i) {
		pthread_create(threads + 2 * i, NULL, process_delta_queue, delta_stmt_thread[i]);
	}

	for (i = 0; i < ncpu; ++i) {
		pthread_create(threads + 2 * i + 1, NULL, process_compress_queue, update_stmt_thread[i]);
	}

	printf("Importing revisions...\n0.0%%");

	while ((todo = branch_todo) != NULL) {
		branch_todo = todo->next;
		process_branch(todo->branch_id, 0);
		free(todo);
	}

	if (sqlite3_prepare(db, "SELECT b.id FROM branches b "
	    "WHERE b.vendorbranch", -1, &vendor_stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	while (sqlite3_step(vendor_stmt) == SQLITE_ROW)
		process_branch(sqlite3_column_int64(vendor_stmt, 0), 1);
	sqlite3_finalize(vendor_stmt);

	printf("\r...done\n");
	fflush(stdout);

	flush_pending_manifest();

	free(manifest_digest);
	for (j = 0; j < files; ++j)
		free(manifest_path[j]);
	free(manifest_file_map);
	free(manifest_path);
	free(cur_last_seen);
	free(last_author);

	if (sqlite3_prepare(outdb, "SELECT uuid FROM blob WHERE content ISNULL",
	    -1, &stmt, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	if (sqlite3_prepare(db, "SELECT content FROM content WHERE hash=?",
	    -1, &stmt2, NULL))
		errx(1, "sqlite3_prepare failed: %s", sqlite3_errmsg(db));
	terminate = 1;
	pthread_mutex_lock(&delta_queue_mutex);
	pthread_cond_broadcast(&delta_queue_busy);
	pthread_mutex_unlock(&delta_queue_mutex);

	for (i = 0; i < ncpu; ++i) {
		pthread_join(threads[2 * i], NULL);
	}

	terminate = 2;
	pthread_mutex_lock(&compress_queue_mutex);
	pthread_cond_broadcast(&compress_queue_busy);
	pthread_mutex_unlock(&compress_queue_mutex);

	for (i = 0; i < ncpu; ++i) {
		pthread_join(threads[2 * i + 1], NULL);
	}

	for (i = 0; i < ncpu; ++i) {
		sqlite3_finalize(update_stmt_thread[i]);
		sqlite3_finalize(delta_stmt_thread[i]);
	}

	printf("Copying remaining revision blobs...\n");
	while (sqlite3_step(stmt) == SQLITE_ROW) {
		if (sqlite3_bind_text(stmt2, 1,
		    (const char *)sqlite3_column_text(stmt, 0),
		    -1, SQLITE_STATIC))
			errx(1, "sqlite3_bind_text failed");
		if (sqlite3_step(stmt2) != SQLITE_ROW)
			errx(1, "missing blob %s",
			sqlite3_column_text(stmt, 0));

		if (sqlite3_bind_blob(update_stmt_main, 1,
		    sqlite3_column_blob(stmt2, 0),
		    sqlite3_column_bytes(stmt2, 0), SQLITE_STATIC))
			errx(1, "sqlite3_bind_blob failed");
		if (sqlite3_bind_text(update_stmt_main, 2,
		    (const char *)sqlite3_column_text(stmt, 0), -1,
		    SQLITE_STATIC))
			errx(1, "sqlite3_bind_text_failed");
		if (sqlite3_step(update_stmt_main) != SQLITE_DONE)
			errx(1, "sqlite3_step for delta update failed");
		sqlite3_reset(update_stmt_main);
		sqlite3_reset(stmt2);
	}

	sqlite3_finalize(stmt2);
	sqlite3_finalize(stmt);


	sqlite3_finalize(delta_insert_stmt_main);
	sqlite3_finalize(update_stmt_main);
	sqlite3_finalize(insert_stmt);
	sqlite3_finalize(check_uuid_stmt);

	free(threads);
	free(update_stmt_thread);
	free(delta_stmt_thread);

	if (sqlite3_exec(outdb, "COMMIT", NULL, 0, NULL))
		errx(1, "commit failed");

	if (sqlite3_close(outdb))
		errx(1, "db close failed");

	if (sqlite3_exec(db, "COMMIT", NULL, 0, NULL))
		errx(1, "commit failed");

	if (sqlite3_close(db))
		errx(1, "db close failed: %s", sqlite3_errmsg(db));

	return 0;
}
