/*-
 * Copyright (c) 2010 Joerg Sonnenberger <joerg@NetBSD.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <assert.h>
#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <sqlite3.h>
#include <string.h>

#include "common.h"

static const char tag_branch_statement[] =
    "SELECT s.symbol, f.path, s.revision FROM branches b, symbol s, file f "
    "WHERE b.symbol=s.symbol AND s.file=f.id AND NOT s.branch "
    "ORDER BY s.symbol, f.path";

static const char vendor_branch_statement[] =
    "SELECT s.symbol, f.path, s.revision FROM branches b, symbol s, file f "
    "WHERE b.vendorbranch AND b.symbol=s.symbol AND s.file=f.id AND s.branch "
    "AND s.revision != \"1.1.1\" "
    "ORDER BY s.symbol, f.path";

static const char bad_default_branch_statement[] =
    "SELECT path, branch FROM file WHERE branch NOTNULL AND branch!=\"1.1.1\" "
    "ORDER BY path";

static const char default_branch_statement[] =
    "SELECT f.path FROM file f WHERE f.branch NOTNULL AND NOT EXISTS"
    "(SELECT * FROM symbol s WHERE s.file=f.id AND "
    "s.branch AND s.revision=\"1.1.1\") ORDER BY f.path";

static const char no_default_branch_statement[] =
    "SELECT f.path FROM file f WHERE f.branch ISNULL AND NOT EXISTS"
    "(SELECT * FROM revision r WHERE r.file=f.id AND"
    " r.revision >= \"1.2\" AND r.revision NOT GLOB \"1.*.*\") "
    "AND EXISTS (SELECT * FROM symbol s WHERE s.file=f.id AND "
    "s.branch AND s.revision=\"1.1.1\") AND EXISTS (SELECT * FROM revision "
    "WHERE file=f.id AND revision=\"1.1\" AND message=\"Initial revision\n\") "
    "ORDER BY f.path";

static const char disconnected_statement[] =
    "SELECT f.path, r.revision FROM file f, revision r WHERE "
    "f.id=r.file AND r.id NOT IN (SELECT revision FROM revbranches) "
    "ORDER BY f.path";

/*
 * Time stamps are supposed to increase monotony.
 *
 * Exceptions:
 * - the first revision of a branch has the same time as 1.1 if the file
 *   was added on the branch
 * - the first revision of a branch has the same time as 1.1 if the file
 *   was added after the branch was initially cut
 * - the first revision was added by cvs import after having added normally.
 */
static const char backwards_statement[] =
    "SELECT f.path, r1.revision, r2.revision "
    "FROM file f, revision r1, revision r2, revision_link l, "
    "revbranches b1, revbranches b2 WHERE "
    "f.id=r1.file AND l.parent=r1.id AND l.child=r2.id AND "
    "b1.revision=r1.id AND b2.revision=r2.id AND "
    "((r1.date>r2.date AND r2.revision != \"1.1.1.1\") OR "
    "(r1.date=r2.date AND b1.branch=b2.branch) OR "
    "(r1.date=r2.date AND b1.branch!=b2.branch"
    " AND r2.revision != \"1.1.1.1\" AND r1.content NOTNULL"
    " AND r2.content NOT NULL))";

static void
usage(void)
{
	fprintf(stderr, "Usage: %s database", getprogname());
	exit(2);
}

static int got_warning, got_disconnected, got_backwards;
static char *last_branch;

static int
branch_warning(void *cookie, int columns, char **values, char **names)
{
	assert(columns == 3);

	if (last_branch == NULL || strcmp(last_branch, values[0])) {
		free(last_branch);
		if ((last_branch = strdup(values[0])) == NULL)
			errx(2, "strdup failed");
		if (got_warning)
			putchar('\n');
		printf(cookie,
		    values[0]);
	}
	got_warning = 1;
	printf("%s %s\n", values[1], values[2]);
	return 0;
}

static int
bad_defbranch_warning(void *cookie, int columns, char **values, char **names)
{
	static int got_bad_defbranch_warning;

	assert(columns == 2);
	if (!got_bad_defbranch_warning) {
		if (got_warning)
			putchar('\n');
		printf("Files with bad default branch\n");
		got_warning = 1;
		got_bad_defbranch_warning = 1;
	}
	printf("%s %s\n", values[0], values[1]);
	return 0;
}

static int
defbranch_warning(void *cookie, int columns, char **values, char **names)
{
	static int got_defbranch_warning;

	assert(columns == 1);
	if (!got_defbranch_warning) {
		if (got_warning)
			putchar('\n');
		printf("Files with default branch, but no vendor branch\n");
		got_warning = 1;
		got_defbranch_warning = 1;
	}
	puts(values[0]);
	return 0;
}

static int
no_defbranch_warning(void *cookie, int columns, char **values, char **names)
{
	static int got_no_defbranch_warning;

	assert(columns == 1);
	if (!got_no_defbranch_warning) {
		if (got_warning)
			putchar('\n');
		printf("Files with missing default branch\n");
		got_warning = 1;
		got_no_defbranch_warning = 1;
	}
	puts(values[0]);
	return 0;
}


static int
disconnected_warning(void *cookie, int columns, char **values, char **names)
{
	assert(columns == 2);

	if (!got_disconnected)
		printf("%sDisconnected revisions\n", got_warning ? "\n" : "");
	got_disconnected = 1;
	got_warning = 1;
	printf("%s %s\n", values[0], values[1]);

	return 0;
}

static int
backwards_warning(void *cookie, int columns, char **values, char **names)
{
	assert(columns == 3);

	if (!got_backwards)
		printf("%sRevisions not newer than previous revision\n",
		    got_warning ? "\n" : "");
	got_backwards = 1;
	got_warning = 1;
	printf("%s %s %s\n", values[0], values[1], values[2]);

	return 0;
}

int
main(int argc, char **argv)
{
	char *errmsg;
	sqlite3 *db;

	if (argc != 2)
		usage();

	if (sqlite3_open_v2(argv[1], &db, SQLITE_OPEN_READWRITE, NULL))
		errx(2, "database open failed: %s", sqlite3_errmsg(db));

	if (sqlite3_exec(db, "PRAGMA cache_size = 16384", NULL, 0, &errmsg))
		errx(2, "raising cache size failed: %s", errmsg);

	if (sqlite3_exec(db, tag_branch_statement, branch_warning,
	    __UNCONST("Conflicting tags for branch %s with revision\n"),
	    &errmsg))
		errx(2, "check for tag/branch mixup failed: %s", errmsg);
	free(last_branch);
	last_branch = NULL;

	if (sqlite3_exec(db, vendor_branch_statement, branch_warning,
	    __UNCONST("Non-vendor branch symbols for %s with revision\n"),
	    &errmsg))
		errx(2, "check for vendor/normal branch mixup failed: %s",
		    errmsg);
	free(last_branch);
	last_branch = NULL;

	if (sqlite3_exec(db, bad_default_branch_statement,
	    bad_defbranch_warning, NULL, &errmsg))
		errx(2, "check for bad default branches failed: %s", errmsg);

	if (sqlite3_exec(db, default_branch_statement, defbranch_warning,
	    NULL, &errmsg))
		errx(2, "check for default branch mixup failed: %s", errmsg);

	if (sqlite3_exec(db, no_default_branch_statement, no_defbranch_warning,
	    NULL, &errmsg))
		errx(2, "check for default branch mixup failed: %s", errmsg);

	if (sqlite3_exec(db, disconnected_statement, disconnected_warning,
	    NULL, &errmsg))
		errx(2, "check for disconnected revisions mixup failed: %s",
		    errmsg);

	if (sqlite3_exec(db, backwards_statement, backwards_warning,
	    NULL, &errmsg))
		errx(2, "check for revisions going backwards failed: %s",
		    errmsg);

	if (sqlite3_close(db))
		errx(2, "db close failed");

	return got_warning;
}
